<?php
/*
 * Instagram Feed Widget
 */
if ( ! class_exists( 'Sanfrancisco_Widget_Instagram_Feed' ) ) {
	class Sanfrancisco_Widget_Instagram_Feed extends WP_Widget {

		function __construct() {

			$widget_opts = array(
				'classname'   => 'sf-widget-instagram-feed',
				'description' => esc_html_x( 'Displays the instagram images', 'Instagram Feed widget description', 'sanfrancisco' )
			);

			parent::__construct( 'sanfrancisco-instagram-feed', sprintf( esc_html_x( '%s - Instagram Feed', 'Instagram Feed widget name', 'sanfrancisco' ), SANFRANCISCO_THEME_TITLE ), $widget_opts );

		}

		function widget( $args, $instance ) {

			echo wp_kses_post( $args['before_widget'] );

			if ( ! empty( $instance['title'] ) ) {
				echo wp_kses_post( $args['before_title'] ) . esc_html( $instance['title'] ) . wp_kses_post( $args['after_title'] );
			}

			if ( ! is_numeric( $instance['number'] ) ) {
				$instance['number'] = 12;
			}

			if ( ! is_numeric( $instance['columns'] ) ) {
				$instance['columns'] = 4;
			}

			//$data = $this->fetch_feed( $instance['username'], $instance['number'] );
			$data = array();

			if ( is_wp_error( $data ) ) {

				echo '<span class="widget-error-text">'. esc_html( $data->get_error_message() ) .'</span>';

			} else { ?>

				<div class="sf-instagram-images clearfix sf-col-<?php echo intval( $instance['columns'] ); ?>">
					<?php foreach ( $data as $k => $v ): ?>
					<a href="<?php echo esc_url( 'https://instagram.com/p/'. $v['code'] .'/' ); ?>" target="_blank" rel="nofollow" class="sf-item">
						<?php $this->image_tag( $v ); ?>
					</a>
					<?php endforeach; ?>
				</div>

			<?php }

			echo wp_kses_post( $args['after_widget'] );

		}

		function image_tag( $obj ) {

			$src = preg_replace( '#\/s[0-9]+x[0-9]+\/#', '/s50x50/', $obj['thumbnail_src'] );
			$attr = array(
				'src'      => 'data:image/gif;base64,R0lGODlhAQABAIAAAAAAAAAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw==',
				'data-src' => esc_url( $src ),
				'class'    => 'sf-instagram-img agg-fade-in lazyloaded',
				'alt'      => array_key_exists( 'caption', $obj ) ? $obj['caption'] : '',
				'width'    => '50',
				'height'   => '50',
				'sizes'    => '(max-width: 750px) 100vw, 750px'
			);

			$srcset = array();
			foreach ( array( '50', '100', '200' ) as $k ) {
				$srcset[] = preg_replace( '#\/s[0-9]+x[0-9]+\/#', '/s'. $k .'x'. $k .'/', $src ) .' '. $k .'w';
			}

			$attr['srcset'] = implode( ', ', $srcset );

			$attr = array_map( 'esc_attr', $attr );

			$image_tag = '<img';
			foreach ( $attr as $name => $value ) {
				$image_tag .= " $name=" . '"' . $value . '"';
			}
			$image_tag .= ' />';

			echo trim( $image_tag );

		}

		function fetch_feed( $username, $slice = 8 ) {

			$remote_url = esc_url( 'http://instagram.com/'. trim( strtolower( $username ) ) );
			$transient_key = 'sanfrancisco_instagram_feed_'. sanitize_title_with_dashes( $username );
			$slice = absint( $slice );

			if ( false === ( $result_data = get_transient( $transient_key ) ) ) {

				$remote = wp_remote_get( $remote_url );

				if ( is_wp_error( $remote ) || 200 != wp_remote_retrieve_response_code( $remote ) ) {
					return new WP_Error( 'not-connected', esc_html__( 'Unable to communicate with Instagram.', 'sanfrancisco' ) );
				}

				preg_match( '#window\.\_sharedData\s\=\s(.*?)\;\<\/script\>#', $remote['body'], $match );

				if ( ! empty( $match ) ) {

					$data = json_decode( end( $match ), true );

					if ( is_array( $data ) && isset ( $data['entry_data']['ProfilePage'][0]['user']['media']['nodes'] ) ) {
						$result_data = $data['entry_data']['ProfilePage'][0]['user']['media']['nodes'];
					}

				}

				if ( is_array( $result_data ) ) {
					set_transient( $transient_key, $result_data, 60 * 60 * 2 );
				}

			}

			if ( empty( $result_data ) ) {
				return new WP_Error( 'no-images', esc_html__( 'Instagram did not return any images.', 'sanfrancisco' ) );
			}

			return array_slice( $result_data, 0, $slice );

		}

		function update( $new_instance, $old_instance ) {

			$instance = $old_instance;

			$instance['title']      = strip_tags( $new_instance['title'] );
			$instance['username']   = strip_tags( $new_instance['username'] );
			$instance['number']     = absint( $new_instance['number'] );
			$instance['columns']     = absint( $new_instance['columns'] );

			return $instance;

		}

		function form( $instance ) {

			$title    = isset( $instance['title'] ) ? $instance['title'] : '';
			$username = isset( $instance['username'] ) ? $instance['username'] : '';
			$number   = isset( $instance['number'] ) ? $instance['number'] : 8;
			$columns   = isset( $instance['columns'] ) ? $instance['columns'] : 4;

			?>
			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'sanfrancisco' ); ?></label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'username' ) ); ?>"><?php esc_html_e( 'Username:', 'sanfrancisco' ); ?></label>
				<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'username' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'username' ) ); ?>" type="text" value="<?php echo esc_attr( $username ); ?>" />
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>"><?php esc_html_e( 'Number of items:', 'sanfrancisco' ); ?></label>
				<input id="<?php echo esc_attr( $this->get_field_id( 'number' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'number' ) ); ?>" type="text" value="<?php echo absint( $number ); ?>" size="3" />
				<span class="sf-widget-form-description sf-inline"><?php echo esc_html_x( '(Max. 12)', 'Max. number of instagram images', 'sanfrancisco' ); ?></span>
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'columns' ) ); ?>"><?php esc_html_e( 'Columns:', 'sanfrancisco' ); ?></label>
				<select id="<?php echo esc_attr( $this->get_field_id( 'columns' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'columns' ) ); ?>">
					<option value="2"<?php echo ( 2 == $columns ) ? ' selected' : ''; ?>>2</option>
					<option value="3"<?php echo ( 3 == $columns ) ? ' selected' : ''; ?>>3</option>
					<option value="4"<?php echo ( 4 == $columns ) ? ' selected' : ''; ?>>4</option>
				</select>
			</p>
			<?php

		}

	}
}