<?php

// Theme Options version definition
define( 'MCO_TO_VERSION', '1.5.1' );

/*
 * Get theme options key for options in db
 */
if ( ! function_exists( 'mco_to_id' ) ) {
	function mco_to_id() {

		return apply_filters( 'mco_to_id', 'mco_to' );

	}
}

/*
 * Register meta box
 */
if ( ! class_exists( 'MCO_Meta_Box' ) ) {
	class MCO_Meta_Box {

		private $meta_box;
		
		private $mco_mb_prefix = 'add_meta';

		function __construct( $meta_box ) {

			if ( ! is_admin() ) {
				return;
			}

			$this->meta_box = $meta_box;

			add_action( $this->mco_mb_prefix . '_boxes', array( &$this, 'mb_amb' ) );
			add_action( 'save_post', array( &$this, 'save_post' ), 1, 2 );

		}

		function mb_amb() {

			$cb_fields = array_key_exists( 'fields', $this->meta_box ) ? $this->meta_box['fields'] : array();

			foreach ( $this->meta_box['pages'] as $page ) {

				call_user_func_array( $this->mco_mb_prefix . '_box', array(
					$this->meta_box['id'],
					$this->meta_box['title'],
					$this->meta_box['render'],
					$page,
					$this->meta_box['context'],
					$this->meta_box['priority'],
					$cb_fields
				) );

				if ( array_key_exists( 'class', $this->meta_box ) && is_array( $this->meta_box['class'] ) ) {
					$mco_metabox_class = $this->meta_box['class'];
					add_filter( 'postbox_classes_'. $page .'_'. $this->meta_box['id'], function( $classes ) use( &$mco_metabox_class)  {
						if ( is_array( $mco_metabox_class ) ) {
							foreach ( $mco_metabox_class as $class ) {
								$classes[] = $class;
							}
						}
						return $classes;
					} );
				}

			}

		}

		function save_post( $post_id, $post_obj ) {

			global $pagenow;

			/* don't save if $_POST is empty */
			if ( empty( $_POST ) || ( array_key_exists( 'vc_inline', $_POST ) && $_POST['vc_inline'] == true ) ) {
				return $post_id;
			}

			/* don't save during quick edit */
			if ( $pagenow == 'admin-ajax.php' ) {
				return $post_id;
			}

			/* don't save during autosave */
			if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
				return $post_id;
			}

			/* don't save if viewing a revision */
			if ( $post_obj->post_type == 'revision' || $pagenow == 'revision.php' ) {
				return $post_id;
			}

			/* verify nonce */
			if ( ! array_key_exists( $this->meta_box['id'] .'_nonce', $_POST ) || ! wp_verify_nonce( $_POST[ $this->meta_box['id'] . '_nonce'], $this->meta_box['id'] ) ) {
				return $post_id;
			}

			/* check permissions */
			if ( array_key_exists( 'post_type', $_POST ) && 'page' == $_POST['post_type'] ) {
				if ( ! current_user_can( 'edit_page', $post_id ) ) {
					return $post_id;
				}
			} else {
				if ( ! current_user_can( 'edit_post', $post_id ) ) {
					return $post_id;
				}
			}

			if ( array_key_exists( 'save', $this->meta_box ) ) {

				call_user_func_array( $this->meta_box['save'], array( $post_obj, $this->meta_box ) );

			} else if ( array_key_exists( 'fields', $this->meta_box ) && is_array( $this->meta_box['fields'] ) && ! empty( $this->meta_box['fields'] ) ) {

				$values = MCO_Theme_Options::sanitize_fields( $_POST, $this->meta_box['fields'] );

				foreach ( $values as $k => $v ) {

					if ( $v != get_post_meta( $post_id, $k, true ) ) {
						update_post_meta( $post_id, $k, $v );
					} elseif ( $v == '' ) {
						delete_post_meta( $post_id, $k, get_post_meta( $post_id, $k, true ) );
					}

				}

			}

			return $post_id;

		}

	}
}

/*
 * Build meta boxes & page builder & theme options page
 */
if ( ! class_exists( 'MCO_Theme_Options' ) ) {
	class MCO_Theme_Options {

		private $pb_settings = array();
		private $options = array();
		private $to_settings = array();
		private $to_saved = array();
		private $term_fields = array();
		private $widget_fields = array();

		public function __construct( $options = array() ) {

			if ( array_key_exists( 'lib_uri', $options ) ) {
				$options['lib_uri'] = trailingslashit( $options['lib_uri'] );
			}

			if ( array_key_exists( 'lib_path', $options ) ) {
				$options['lib_path'] = trailingslashit( $options['lib_path'] );
			}

			$this->options = $options;

			if ( $this->screen_belongs_to_theme_options() ) {

				/* enqueue admin scripts & styles */
				add_action( 'admin_enqueue_scripts', array( &$this, 'enqueue_scripts' ), 99 );

				/* move all "advanced" metaboxes above the default editor */
				add_action( 'edit_form_after_title', array( &$this, 'move_advanced_meta_boxes' ) );

				/* preload wp editor */
				add_action( 'admin_footer', array( $this, 'preload_wp_editor' ) );

			}

		}

		/*
		 * Check if current screen belongs to theme options
		 */
		public function screen_belongs_to_theme_options() {

			global $pagenow;

			$white_label = array(
				'post.php',
				'post-new.php',
				'term.php',
				'edit-tags.php',
				'widgets.php'
			);

			if ( ( $pagenow == 'themes.php'
				&& array_key_exists( 'page', $_GET )
				&& $_GET['page'] == 'mco-theme-options' ) || in_array( $pagenow, $white_label ) ) {
				return true;
			}

			return false;

		}

		/*
		 * Admin Scripts & Styles
		 */
		public function enqueue_scripts( $hook ) {

			$enqueue = false;

			$hook_arr = array(
				'widgets.php',
				'post.php',
				'post-new.php',
				'term.php',
				'edit-tags.php',
				'appearance_page_mco-theme-options'
			);

			if ( in_array( $hook, $hook_arr ) ) {
				$enqueue = true;
			}

			if ( $hook == 'post.php' ) {
				global $post;
				$post_type = $post->post_type;
			} else if ( array_key_exists( 'post_type', $_GET ) ) {
				$post_type = $_GET['post_type'];
			}

			if ( isset( $post_type ) && ! in_array( $post_type, array( 'post', 'page', 'product' ) ) ) {
				$enqueue = false;
			}

			if ( $enqueue ) {

				/* load open sans google font */
				wp_enqueue_style( 'google-font-open-sans', '//fonts.googleapis.com/css?family=Open+Sans:300,400,400i,700,800' );

				/* load fontawesome */
				wp_enqueue_style( 'font-awesome-4.7.0', $this->options['lib_uri'] .'lib/font-awesome-4.7.0/css/font-awesome.min.css', array(), '4.7.0' );

				/* load select2-css */
				wp_enqueue_style( 'select2-css', $this->options['lib_uri'] .'lib/select2/select2.min.css', array(), '4.0.3' );

				/* load main css */
				wp_enqueue_style( 'mco-theme-options-css', $this->options['lib_uri'] .'css/mco-theme-options.css', array(), MCO_TO_VERSION );

				/* load Spectrum (alternative to WP colorpicker) */
				wp_enqueue_style( 'spectrum-css', $this->options['lib_uri'] .'lib/spectrum/spectrum.css' );

				/* WP Media Uploader */
				wp_enqueue_media();

				/* load jQuery-ui slider */
				wp_enqueue_script( 'jquery-ui-slider' );

				/* load jQuery-ui sortable */
				wp_enqueue_script( 'jquery-ui-sortable' );

				/* load Spectrum (alternative to WP colorpicker) */
				wp_enqueue_script( 'spectrum-js', $this->options['lib_uri'] .'lib/spectrum/spectrum.js' );

				/* load bootstrap.js */
				wp_enqueue_script( 'bootstrap-js', $this->options['lib_uri'] .'lib/bootstrap-3.3.6/js/bootstrap.js' );

				/* load select2-js */
				wp_enqueue_script( 'select2-js', $this->options['lib_uri'] .'lib/select2/select2.full.min.js', array(), '4.0.3', true );

				/* load ace editor */
				wp_enqueue_script( 'ace-code-highlighter-js', $this->options['lib_uri'] .'lib/ace/ace.js', array( 'jquery' ), false, true );

				/* load main js */
				wp_enqueue_script( 'mco-theme-options-js', $this->options['lib_uri'] .'js/mco-theme-options.js', array( 'jquery' ), MCO_TO_VERSION, true );

				$mco_localized_arr = array(
					'add_new_section'            => esc_html__( 'Add New Page Section', 'sanfrancisco' ),
					'new_section'                => esc_html__( 'New Page Section', 'sanfrancisco' ),
					'section_settings'           => esc_html__( 'Page Section Settings', 'sanfrancisco' ),
					'add_new_module'             => esc_html__( 'Add New Module', 'sanfrancisco' ),
					'new_module'                 => esc_html__( 'New Module', 'sanfrancisco' ),
					'module_settings'            => esc_html__( 'Module Settings', 'sanfrancisco' ),
					'select_or_upload_image'     => esc_html__( 'Select or Upload an Image', 'sanfrancisco' ),
					'select_or_upload_images'    => esc_html__( 'Select or Upload Images', 'sanfrancisco' ),
					'use_this_image'             => esc_html__( 'Use This Image', 'sanfrancisco' ),
					'use_this_images'            => esc_html__( 'Use These Images', 'sanfrancisco' ),
					'select_or_upload_video'     => esc_html__( 'Select or Upload a Video', 'sanfrancisco' ),
					'use_this_video'             => esc_html__( 'Use This Video', 'sanfrancisco' ),
					'select_or_upload_audio'     => esc_html__( 'Select or Upload an Audio', 'sanfrancisco' ),
					'use_this_audio'             => esc_html__( 'Use This Audio', 'sanfrancisco' ),
					'page_section'               => esc_html__( 'Page Section', 'sanfrancisco' ),
					'press_ok_to_delete_section' => esc_html__( 'Press OK to delete section, Cancel to leave', 'sanfrancisco' ),
					'press_ok_to_delete_module'  => esc_html__( 'Press OK to delete module, Cancel to leave', 'sanfrancisco' ),
					'press_ok_to_delete_sidebar' => esc_html__( 'Press OK to delete sidebar, Cancel to leave', 'sanfrancisco' ),
					'confirm_to_leave'           => esc_html__( 'The changes you made will be lost if you navigate away from this page.', 'sanfrancisco' ),
					'remove_module'              => esc_html__( 'Remove', 'sanfrancisco' ),
					'edit_module'                => esc_html__( 'Edit', 'sanfrancisco' ),
					'changes_unsaved'            => esc_html__( 'Settings have changed, you should save them!', 'sanfrancisco' ),
					'confirm_reset_section'      => esc_html__( 'Press OK to reset options in this section, Cancel to leave', 'sanfrancisco' ),
					'confirm_reset_all'          => esc_html__( 'Press OK to reset all options, Cancel to leave', 'sanfrancisco' ),
					'fonts_map'                  => $this->get_fonts_map()
				);

				wp_localize_script( 'mco-theme-options-js', 'mcoI18nLocale', $mco_localized_arr );

			}

		}

		/*
		 * Preload wp editor in order to fix the issue on ajax call
		 */
		public function preload_wp_editor() {

			echo '<div style="display: none;">';

			wp_editor( '', 'mco_preloaded_editor_id', array(
				'textarea_name'    => 'mco_preloaded_editor_name',
				'textarea_rows'    => 20,
				'editor_class'     => 'mco-form-control',
				'drag_drop_upload' => true
			) );

			echo '</div>';

		}

		/*
		 * Initialize theme options page
		 */
		public function theme_options( $args ) {

			$this->to_settings = $args;
			$this->to_saved = get_option( mco_to_id(), array() );

			if ( ! is_array( $this->to_saved ) ) {
				$this->to_saved = array();
			}

			/* add pages & menu items */
			add_action( 'admin_menu', array( $this, 'add_theme_options_page' ), 10 );

			/* save theme options */
			add_action( 'wp_ajax_save_theme_options', array( $this, 'save_theme_options' ) );
			add_action( 'wp_ajax_nopriv_save_theme_options', array( $this, 'save_theme_options' ) );

			/* reset theme options */
			add_action( 'wp_ajax_reset_theme_options', array( $this, 'reset_theme_options' ) );
			add_action( 'wp_ajax_nopriv_reset_theme_options', array( $this, 'reset_theme_options' ) );

		}

		/*
		 * Add theme options menu item to the admin menu
		 */
		public function add_theme_options_page() {

			$type_1 = 'submenu';
			$pg_1 = 'page';

			$theme_check_bs   = "add_{$type_1}_{$pg_1}";

			$theme_check_bs(
				'themes.php',
				$this->to_settings['title'],
				$this->to_settings['title'],
				'edit_theme_options',
				'mco-theme-options',
				array( $this, 'render_options_ui' )
			);

		}

		/*
		 * Save theme options
		 */
		public function save_theme_options() {

			/* verify nonce */
			/*if ( ! array_key_exists( $this->to_settings['id'] .'_nonce', $_POST ) || ! wp_verify_nonce( $_POST[ $this->to_settings['id'] . '_nonce'], $this->to_settings['id'] ) ) {
				return;
			}*/

			$this->to_saved = self::sanitize_fields( $_POST, $this->to_settings['fields'] );

			do_action( 'mco_before_save_theme_options', $this->to_saved );

			update_option( mco_to_id(), $this->to_saved );

			do_action( 'mco_after_save_theme_options', $this->to_saved );
		
			$this->render_options_ui( null, array(
				'active_section' => array_key_exists( 'active_section', $_POST ) ? $_POST['active_section'] : false,
				'active_tab' => array_key_exists( 'active_tab', $_POST ) ? $_POST['active_tab'] : false
			) );

			exit;

		}

		/*
		 * Reset theme options
		 */
		public function reset_theme_options() {

			/* verify nonce */
			if ( ! array_key_exists( $this->to_settings['id'] .'_nonce', $_POST ) || ! wp_verify_nonce( $_POST[ $this->to_settings['id'] . '_nonce'], $this->to_settings['id'] ) ) {
				return;
			}

			$reset_type = ( array_key_exists( 'reset_type', $_POST ) && $_POST['reset_type'] == 'section' ) ? 'section' : 'all';
			$active_section = array_key_exists( 'active_section', $_POST ) ? $_POST['active_section'] : false;

			$section_fields = ( $reset_type == 'all' );
			foreach ( $this->to_settings['fields'] as $v ) {

				if ( $v['type'] == 'section' && $reset_type == 'section' ) {
					$section_fields = ( $v['id'] == $active_section );
				}

				if ( ! in_array( $v['type'], array( 'section', 'tab' ) ) && $section_fields ) {
					if ( array_key_exists( 'std', $v ) ) {
						$this->to_saved[ $v['id'] ] = $v['std'];
					} else if ( array_key_exists( $v['id'], $this->to_saved ) ) {
						unset( $this->to_saved[ $v['id'] ] );
					}
				}

			}

			update_option( mco_to_id(), $this->to_saved );

			$this->render_options_ui( null, array(
				'active_section' => $active_section,
				'active_tab' => array_key_exists( 'active_tab', $_POST ) ? $_POST['active_tab'] : false
			) );

			exit;

		}

		/*
		 * Get new instance of meta box
		 */
		public function meta_box( $args ) {

			return new MCO_Meta_Box( $args );

		}

		/*
		 * Initialize page builder
		 */
		public function page_builder( $args ) {

			if ( ! is_array( $args ) || ! array_key_exists( 'id', $args ) || ! array_key_exists( 'settings', $args ) ) {
				return;
			}

			$this->pb_settings = $args['settings'];

			if ( ! array_key_exists( 'title', $args ) ) {
				$args['title'] = '';
			}

			if ( ! array_key_exists( 'display', $args ) ) {
				$args['display'] = 'page';
			}

			if ( ! array_key_exists( 'context', $args ) ) {
				$args['context'] = 'normal';
			}

			if ( ! array_key_exists( 'priority', $args ) ) {
				$args['priority'] = 'high';
			}

			$args['pages']  = is_array( $args['display'] ) ? $args['display'] : array( $args['display'] );
			$args['fields'] = $this->pb_settings;
			$args['render'] = array( $this, 'render_page_builder' );
			$args['save']   = array( $this, 'save_page_builder' );
			$args['class']  = $this->postbox_class( $args );

			$this->meta_box( $args );

			add_action( 'wp_ajax_edit_section', array( &$this, 'get_pb_fields' ) );
			add_action( 'wp_ajax_edit_module', array( &$this, 'get_pb_fields' ) );
			add_action( 'admin_footer', array( &$this, 'pb_modal' ) );

		}

		/*
		 * Build page builder meta box
		 */
		public function render_page_builder( $post, $metabox ) {

			$mco_values = get_post_meta( $post->ID, $metabox['id'], true );
			$mco_fields = array_key_exists( 'args', $metabox ) ? $metabox['args'] : array();

			$mco_labels = array();
			foreach ( array( 'section_settings', 'section_modules' ) as $i ) {
				if ( array_key_exists( $i, $mco_fields ) && is_array( $mco_fields[ $i ] ) ) {
					foreach ( $mco_fields[ $i ] as $k ) {
						if ( array_key_exists( 'choices', $k ) ) {
							foreach ( $k['choices'] as $v ) {
								if ( array_key_exists( 'label', $v ) && ( ! array_key_exists( 'type', $v ) || $v['type'] != 'optgroup' ) ) {
									$mco_labels[ $v['value'] ] = $v['label'];
								} else if ( array_key_exists( 'text', $v ) ) {
									$mco_labels[ $v['value'] ] = $v['text'];
								} else if ( array_key_exists( 'alt', $v ) ) {
									$mco_labels[ $v['value'] ] = $v['alt'];
								}
							}
						}
					}
				}
			}

			echo '<input type="hidden" name="'. esc_attr( $metabox['id'] ) .'_nonce" value="' . esc_attr( wp_create_nonce( $metabox['id'] ) ) . '" />';

			?>
			<input type="hidden" name="mco_pb_id" id="mco_pb_id" value="<?php echo esc_attr( $metabox['id'] ); ?>" />
			<div class="mco-container mco-pb">
				<div class="mco-btn-wrapper">
					<button type="button" class="mco-btn mco-btn-lg mco-btn-green mco-btn-add btn-pb-edit" data-type="section">
						<span class="fa fa-plus-square"></span> <?php esc_html_e( 'Add New Page Section', 'sanfrancisco' ); ?>
					</button>
				</div>
					<div class="mco-section-wrapper">
					<?php if ( is_array( $mco_values ) ): foreach ( $mco_values as $section_uid => $section ):

						$mco_label = array_key_exists( $section['settings']['layout'], $mco_labels ) ? $mco_labels[ $section['settings']['layout'] ] : $section['settings']['layout'];

						?>
						<div class="mco-section mco-section-<?php echo esc_attr( $section_uid ); ?>">

							<div class="mco-section-header clearfix">
								<h2 class="mco-title">
									<span class="fa fa-bars"></span>
									<span class="title"><?php echo esc_html( $mco_label ); ?></span>
								</h2>
								<ul class="mco-section-settings">
									<li><button type="button" class="mco-btn btn-pb-edit" data-type="section" data-section-uid="<?php echo esc_attr( $section_uid ); ?>"><span class="fa fa-edit"></span></button></li>
									<li><button type="button" class="mco-btn btn-pb-remove" data-type="section" data-section-uid="<?php echo esc_attr( $section_uid ); ?>"><span class="fa fa-times"></span></button></li>
								</ul>
							</div>

							<div class="mco-section-body">

								<ul class="mco-section-modules">
									<?php
									if ( array_key_exists( 'modules', $section ) && is_array( $section['modules'] ) ):
										foreach ( $section['modules'] as $module_uid => $module ):

											$layout_name = array_key_exists( $module['layout'], $mco_labels ) ? $mco_labels[ $module['layout'] ] : $module['layout'];
											$module_title = array_key_exists( 'title', $module ) ? $module['title'] : '';

											?>
											<li class="mco-module mco-module-<?php echo esc_attr( $module_uid ); ?>">

												<div class="mco-module-header">
													<h3 class="mco-title">
														<span class="fa fa-angle-double-right"></span>
														<?php if ( ! empty( $module_title ) ): ?>
														<span class="name"><?php echo esc_html( $module_title ); ?></span>
														<?php endif; ?>
														<span class="subline"><?php echo esc_html( $layout_name ); ?></span>
													</h3>
												</div>

												<ul class="mco-module-settings">
													<li><button type="button" class="mco-btn btn-pb-edit" data-type="module" data-section-uid="<?php echo esc_attr( $section_uid ); ?>" data-module-uid="<?php echo esc_attr( $module_uid ); ?>"><span class="fa fa-edit"></span><?php esc_html_e( 'Edit', 'sanfrancisco' ); ?></button></li>
													<li><button type="button" class="mco-btn btn-pb-remove" data-type="module" data-section-uid="<?php echo esc_attr( $section_uid ); ?>" data-module-uid="<?php echo esc_attr( $module_uid ); ?>"><span class="fa fa-times"></span><?php esc_html_e( 'Remove', 'sanfrancisco' ); ?></button></li>
												</ul>

												<input type="hidden" id="mco-module-<?php echo esc_attr( $module_uid ); ?>-values" name="<?php echo esc_attr( $metabox['id'] .'['. $section_uid .'][modules]['.$module_uid .']' ); ?>" value="<?php echo esc_attr( json_encode( $module ) ); ?>" />

											</li>
											<?php
										endforeach;
									endif;
									?>
								</ul>

								<div class="mco-btn-wrapper">
									<button type="button" class="mco-btn mco-btn-red mco-btn-add btn-pb-edit" data-type="module" data-section-uid="<?php echo esc_attr( $section_uid ); ?>">
										<span class="fa fa-plus-square"></span> <?php esc_html_e( 'Add New Module', 'sanfrancisco' ); ?>
									</button>
								</div>

							</div>

							<input type="hidden" id="mco-section-<?php echo esc_attr( $section_uid ); ?>-values" name="<?php echo esc_attr( $metabox['id'] .'['. $section_uid .'][settings]' ) ?>" value="<?php echo esc_attr( json_encode( $section['settings'] ) ); ?>" />

						</div>
					<?php endforeach; endif; ?>
					</div>
			</div>
			<?php

		}

		/*
		 * Save page builder meta box on post save
		 */
		public function save_page_builder( $post, $metabox ) {

			if ( ! is_array( $metabox ) || ! array_key_exists( 'fields', $metabox ) || ! is_array( $metabox['fields'] ) || empty( $metabox['fields'] ) ) {
				return;
			}

			$old_value = get_post_meta( $post->ID, $metabox['id'], true );
			$new_value = array();

			if ( array_key_exists( 'section_settings', $metabox['fields'] ) && array_key_exists( $metabox['id'], $_POST ) && is_array( $_POST[ $metabox['id'] ] ) ) {

				foreach ( $_POST[ $metabox['id'] ] as $section_uid => $section ) {

					if ( ! array_key_exists( 'settings', $section ) && ! is_string( $section['settings'] ) ) {
						continue;
					}

					$section['settings'] = json_decode( wp_unslash( $section['settings'] ), true );

					if ( ! is_array( $section['settings'] ) ) {
						continue;
					}

					$new_value[ $section_uid ] = array(
						'id'       => $section_uid,
						'settings' => MCO_Theme_Options::sanitize_fields( $section['settings'], $metabox['fields']['section_settings'] )
					);

					if ( array_key_exists( 'modules', $section ) && is_array( $section['modules'] ) && array_key_exists( 'section_modules', $metabox['fields'] ) && is_array( $metabox['fields']['section_modules'] ) ) {
						$new_value[ $section_uid ]['modules'] = array();
						foreach ( $section['modules'] as $module_uid => $module ) {
							$module = json_decode( wp_unslash( $module ), true );
							if ( is_array( $module ) ) {
								$new_value[ $section_uid ]['modules'][ $module_uid ] = MCO_Theme_Options::sanitize_fields( $module, $metabox['fields']['section_modules'] );
								$new_value[ $section_uid ]['modules'][ $module_uid ]['id'] = $module_uid;
							}
						}
					}

				}

			}

			if ( $new_value !== $old_value ) {
				update_post_meta( $post->ID, $metabox['id'], $new_value );
			} else if ( '' == $new_value && $old_value ) {
				delete_post_meta( $post->ID, $metabox['id'], $old_value );
			}

		}

		/*
		 * Get section or module form elements for page builder
		 */
		public function get_pb_fields() {

			if ( ! is_array( $this->pb_settings ) || empty( $this->pb_settings ) || ! array_key_exists( 'section_settings', $this->pb_settings ) ) {
				esc_html_e( 'Misconfigurated section settings. Please, contact with the theme author.', 'sanfrancisco' );
			}

			$mco_type = ( $_POST['action'] == 'edit_section' ) ? 'section' : 'module';
			$mco_values = ( array_key_exists( 'values', $_POST ) && ! empty( $_POST['values'] ) ) ? $_POST['values'] : false;

			$mco_key = ( $mco_type == 'module' ) ? 'section_modules' : 'section_settings';

			echo '<div class="mco-container">';

			$this->render_fields( $this->pb_settings[ $mco_key ], $mco_values );

			echo '</div>';

			exit;

		}

		/*
		 * Put an instance of modal dialog into the footer
		 */
		public function pb_modal() {

			?>
			<div class="mco-modal mco-pb-modal fade" id="mco-pb-modal" tabindex="-1" role="dialog" aria-labelledby="mco-pb-modal-title" aria-hidden="true">
				<div class="mco-modal-dialog modal-dialog">
					<div class="mco-modal-content">
						<div class="mco-modal-header">
							<a href="#" class="mco-close mco-btn" data-dismiss="modal" aria-hidden="true"><span class="fa fa-times"></span></a>
							<h3 class="mco-modal-title"><?php esc_html_e( 'Section Settings', 'sanfrancisco' ); ?></h3>
						</div>
						<div class="mco-modal-body">
							<?php esc_html_e( 'Loading...', 'sanfrancisco' ); ?>
						</div>
						<div class="mco-modal-footer">
							<button type="button" class="mco-btn mco-btn-cancel" data-dismiss="modal" aria-hidden="true"><?php esc_html_e( 'Cancel', 'sanfrancisco' ); ?></button>
							<button type="button" class="mco-btn mco-btn-green mco-btn-save"><?php esc_html_e( 'Save Changes', 'sanfrancisco' ); ?></button>
						</div>
					</div>
				</div>
			</div>
			<?php

		}

		/*
		 * Initialize widget options fields
		 */
		public function widget_options( $args ) {

			$this->widget_fields[ $args['id'] ] = $args['fields'];

			add_action( 'in_widget_form', array( &$this, 'widget_fields' ), 10, 3 );
			add_filter( 'widget_update_callback', array( &$this, 'save_widget' ), 10, 4 );

		}

		/*
		 * Add extra form fields to the widget
		 */
		public function widget_fields( $widget, $return, $instance ) {

			if ( array_key_exists( $widget->id_base, $this->widget_fields ) ) {

				$this->render_fields( $this->widget_fields[ $widget->id_base ], $instance, 'widget', array( 'widget' => $widget ) );

			}

		}

		/*
		 * Save widget fields on widget save
		 */
		public function save_widget( $instance, $new_instance, $old_instance, $widget ) {

			if ( array_key_exists( $widget->id_base, $this->widget_fields ) ) {

				$values = wp_parse_args( self::sanitize_fields( $new_instance, $this->widget_fields[ $widget->id_base ] ), $instance );

				foreach ( $values as $k => $v ) {
					$new_instance[ $k ] = $v;
				}

				return $new_instance;

			}

			return $instance;

		}

		/*
		 * Initialize term options fields
		 */
		public function term_options( $args ) {

			$this->term_fields[ $args['taxonomy'] ] = $args['fields'];

			add_action( $args['taxonomy'] .'_add_form_fields', array( &$this, 'term_add_form_fields' ) );
			add_action( $args['taxonomy'] .'_edit_form_fields', array( &$this, 'term_edit_form_fields' ), 10, 2 );
			add_action( 'edited_'. $args['taxonomy'], array( &$this, 'save_term' ), 10, 2 );
			add_action( 'create_'. $args['taxonomy'], array( &$this, 'save_term' ), 10, 2 );

		}

		/*
		 * Add extra form fields to the term edit page
		 */
		public function term_edit_form_fields( $term, $taxonomy ) {

			$values = array();
			foreach ( $this->term_fields[ $taxonomy ] as $field ) {
				$value = get_term_meta( $term->term_id, $field['id'] );
				if ( ! empty( $value ) ) {
					$values[ $field['id'] ] = $value[0];
				}
			}

			$this->render_fields( $this->term_fields[ $taxonomy ], $values, 'term_edit' );

		}

		/*
		 * Add extra form fields to the term add page
		 */
		public function term_add_form_fields( $taxonomy ) {

			$this->render_fields( $this->term_fields[ $taxonomy ], false, 'term_add' );

		}

		/*
		 * Save term fields on term save
		 */
		public function save_term( $term_id ) {

			$term = get_term( $term_id );

			if ( ! array_key_exists( $term->taxonomy, $this->term_fields ) ) {
				return $term_id;
			}

			$values = self::sanitize_fields( $_POST, $this->term_fields[ $term->taxonomy ] );

			foreach ( $values as $k => $v ) {

				if ( $v != get_term_meta( $term_id, $k, true ) ) {
					update_term_meta( $term_id, $k, $v );
				} elseif ( $v == '' ) {
					delete_term_meta( $term_id, $k, get_term_meta( $term_id, $k, true ) );
				}

			}

			return $term_id;

		}

		/*
		 * Initialize post options meta box
		 */
		public function post_options( $args ) {

			if ( ! is_array( $args ) || ! array_key_exists( 'id', $args ) || ! array_key_exists( 'fields', $args ) ) {
				return;
			}

			if ( ! array_key_exists( 'title', $args ) ) {
				$args['title'] = '';
			}

			if ( ! array_key_exists( 'display', $args ) ) {
				$args['display'] = 'post';
			}

			if ( ! array_key_exists( 'context', $args ) ) {
				$args['context'] = 'normal';
			}

			if ( ! array_key_exists( 'priority', $args ) ) {
				$args['priority'] = 'high';
			}

			$args['pages']  = is_array( $args['display'] ) ? $args['display'] : array( $args['display'] );
			$args['render'] = array( $this, 'render_options_ui' );
			$args['class']  = $this->postbox_class( $args );

			$this->meta_box( $args );

		}

		/*
		 * Build options UI
		 */
		public function render_options_ui( $post = null, $metabox = array() ) {

			global $pagenow;

			$is_to = ( $pagenow == 'themes.php' || ( defined( 'DOING_AJAX' ) && DOING_AJAX ) );

			if ( $is_to ) {
				$fields = $this->to_settings['fields'];
			} else {
				$fields = array_key_exists( 'args', $metabox ) ? $metabox['args'] : array();
			}

			$opt_id = $is_to ? $this->to_settings['id'] : $metabox['id'];
			$sections = $values = array();
			$last_section = $last_tab = $active_section = $active_tab = false;

			if ( $is_to ) {
				$values = $this->to_saved;
				if ( is_array( $metabox ) ) {
					$active_section = array_key_exists( 'active_section', $metabox ) ? $metabox['active_section'] : false;
					$active_tab = array_key_exists( 'active_tab', $metabox ) ? $metabox['active_tab'] : false;
				}
			}

			foreach ( $fields as $k => $v ) {

				if ( $v['type'] == 'section' ) {

					$last_section = $last_tab = $v['id'];
					$sections[ $last_section ] = $v;
					$sections[ $last_section ]['tabs'] = array();

					if ( $active_section && $v['id'] == $active_section ) {
						$sections[ $last_section ]['active'] = true;
					} else if ( array_key_exists( 'active', $v ) && $v['active'] ) {
						$active_section = $v['id'];
					} else {
						$sections[ $last_section ]['active'] = false;
					}

				} else if ( $last_section ) {

					if ( $v['type'] == 'tab' ) {

						$last_tab = $v['id'];
						$sections[ $last_section ]['tabs'][ $last_tab ] = $v;

						if ( $active_tab && $v['id'] == $active_tab ) {
							$sections[ $last_section ]['tabs'][ $last_tab ]['active'] = true;
						} else if ( array_key_exists( 'active', $v ) && $v['active'] ) {
							$active_tab = $v['id'];
						} else {
							$sections[ $last_section ]['tabs'][ $last_tab ]['active'] = false;
						}

					} else {

						if ( ! array_key_exists( $last_tab, $sections[ $last_section ]['tabs'] ) ) {

							$sections[ $last_section ]['tabs'][ $last_tab ] = array(
								'id' => $sections[ $last_section ]['id'],
								'label' => $sections[ $last_section ]['label'],
								'type' => 'tab'
							);

						}

						if ( ! array_key_exists( 'fields', $sections[ $last_section ]['tabs'][ $last_tab ] ) ) {
							$sections[ $last_section ]['tabs'][ $last_tab ]['fields'] = array();
						}

						$sections[ $last_section ]['tabs'][ $last_tab ]['fields'][] = $v;

					}

				}

			}

			if ( ! $last_section ) {
				$sections = array(
					array(
						'id' => 'section',
						'tabs' => array(
							array(
								'id' => 'tab',
								'fields' => $fields
							)
						)
					)
				);
			}

			if ( empty( $sections ) ) {
				return;
			}

			if ( ! $active_section ) {
				foreach ( $sections as $k => $v ) {
					$active_section = $k;
					$sections[ $k ]['active'] = true;
					break;
				}
			}

			if ( ! $active_tab ) {
				foreach ( $sections[ $active_section ]['tabs'] as $k => $v ) {
					$active_tab = $k;
					$sections[ $active_section ]['tabs'][ $k ]['active'] = true;
					break;
				}
			}

			foreach ( $sections as $k => $v ) {

				if ( $k != $active_section ) {
					$c = 0;
					foreach ( $v['tabs'] as $i => $j ) {
						$sections[ $k ]['tabs'][ $i ]['active'] = ( $c == 0 );
						$c++;
					}
				}

				if ( $pagenow == 'post.php' ) {
					foreach ( $v['tabs'] as $t ) {
						foreach ( $t['fields'] as $f ) {
							/** @noinspection PhpUndefinedFieldInspection */
							$value = get_post_meta( $post->ID, $f[ 'id'] );
							if ( ! empty( $value ) ) {
								$values[ $f['id'] ] = $value[0];
							}
						}
					}
				}

			}

			if ( $is_to ) {
				echo '<div id="mco-theme-options-wrapper"><form action="" method="post" id="mco-theme-options-form">';
			}

			echo '<input type="hidden" name="active_section" value="'. esc_attr( $active_section ) .'" />';
			echo '<input type="hidden" name="active_tab" value="'. esc_attr( $active_tab ) .'" />';
			echo '<input type="hidden" name="'. esc_attr( $opt_id ) .'_nonce" value="' . esc_attr( wp_create_nonce( $opt_id ) ) . '" />';

			if ( ! empty( $sections ) ) {

				$has_sections = ( count( $sections ) > 1 );

				?>
				<div class="mco-container mco-settings-wrapper<?php echo ( true == $is_to ) ? ' mco-theme-options' : ''; echo ( true == $has_sections ) ? ' mco-has-sections' : ''; ?>">
					<?php if ( $is_to ): ?>
					<div class="mco-loader">
						<div class="mco-spinner">
							<span class="fa fa-spinner"></span>
						</div>
					</div>
					<?php endif; if ( $has_sections ): ?>
					<div class="mco-nav-tabs">
						<?php if ( $is_to ): ?>
						<div class="mco-brand">
							<p class="mco-title"><?php esc_html_e( 'Theme Options', 'sanfrancisco' ); ?></p>
							<p class="mco-version"><?php echo MCO_TO_VERSION; ?></p>
						</div>
						<?php endif; ?>
						<ul role="tablist">
						<?php
						foreach ( $sections as $k => $v ) {
							$section_active_tab = '';
							foreach ( $v['tabs'] as $i => $j ) {
								if ( $j['active'] ) {
									$section_active_tab = $i;
								}
							}
							?>
							<li class="<?php echo sanitize_html_class( 'mco-nav-tab-'. $v['id'] ) . ( $v['active'] ? ' active' : '' ); ?>" role="presentation">
								<a href="#<?php echo sanitize_key( $v['id'] ); ?>" aria-controls="<?php echo esc_attr( $v['id'] ); ?>" role="tab" data-toggle="tab" aria-expanded="<?php echo ( true == $v['active'] ) ? 'true' : 'false'; ?>" data-active-tab="<?php echo esc_attr( $section_active_tab ); ?>"><?php echo ( array_key_exists( 'icon', $v ) ) ? '<span class="fa fa-'. sanitize_key( $v['icon'] ) .'"></span>' : ''; ?><span class="mco-label"><?php echo esc_html( $v['label'] ); ?></span></a>
							</li>
						<?php } ?>
						</ul>
					</div><!--/.mco-nav-tabs -->
					<?php endif; ?>
					<div class="mco-panels">
						<?php if ( $is_to ): ?>
						<div class="mco-settings-nav">
							<button type="button" class="mco-btn mco-btn-lg mco-btn-submit-to mco-btn-red" data-action="save" data-success-msg="<?php esc_html_e( 'Changes successfully saved!', 'sanfrancisco' ); ?>"><?php esc_html_e( 'Save Changes', 'sanfrancisco' ); ?></button>
							<button type="button" class="mco-btn mco-btn-lg mco-btn-submit-to" data-action="reset" data-type="section" data-success-msg="<?php esc_html_e( 'Section successfully reset!', 'sanfrancisco' ); ?>"><?php esc_html_e( 'Reset Section', 'sanfrancisco' ); ?></button>
							<button type="button" class="mco-btn mco-btn-lg mco-btn-submit-to" data-action="reset" data-type="all" data-success-msg="<?php esc_html_e( 'Theme options successfully reset!', 'sanfrancisco' ); ?>"><?php esc_html_e( 'Reset All', 'sanfrancisco' ); ?></button>
						</div>
						<?php endif; ?>
						<?php foreach ( $sections as $k => $v ) { $has_tabs = ( count( $v['tabs'] ) > 1 ); ?>
						<div role="tabpanel" class="mco-panel fade<?php echo ( true == $v['active'] ) ? ' active in' : ''; ?>" id="<?php echo sanitize_key( $v['id'] ); ?>">
							<?php if ( $has_sections && array_key_exists( 'desc', $v ) ): ?>
							<div class="mco-panel-header">
								<?php if ( array_key_exists( 'icon', $v ) ): ?>
								<span class="fa fa-<?php echo sanitize_key( $v['icon'] ); ?> mco-icon"></span>
								<?php endif; ?>
								<h2 class="mco-title"><?php echo esc_html( $v['label'] ); ?></h2>
								<p class="mco-description"><?php echo esc_html( $v['desc'] ); ?></p>
							</div>
							<?php endif; if ( $has_tabs ): ?>
							<div class="mco-panel-nav-tabs">
								<ul role="tablist">
									<?php foreach ( $v['tabs'] as $j => $t ): $tab_id = $v['id'] .'_'. $t['id']; ?>
									<li role="presentation"<?php echo ( true == $t['active'] ) ? ' class="active"' : ''; ?>>
										<a href="#<?php echo sanitize_key( $tab_id ); ?>" aria-controls="<?php echo sanitize_key( $tab_id ); ?>" role="tab" data-toggle="tab" aria-expanded="<?php echo ( true == $t['active'] ) ? 'true' : 'false'; ?>" data-id="<?php echo esc_attr( $t['id'] ); ?>" data-section="<?php echo esc_attr( $v['id'] ); ?>"><?php echo esc_html( $t['label'] ); ?></a>
									</li>
									<?php endforeach; ?>
								</ul>
							<?php endif; ?>
							<div class="mco-panel-content">
								<?php
								foreach ( $v['tabs'] as $j => $t ): $tab_id = $v['id'] .'_'. $t['id'];
									echo ( true == $has_tabs ) ? '<div role="tabpanel" class="mco-panel-inner fade'. ( $t['active'] ? ' active in' : '' ) .'" id="'. sanitize_key( $tab_id ) .'">' : '';
									if ( array_key_exists( 'fields', $t ) ) {
										$this->render_fields( $t['fields'], $values, 'horizontal', array( 'all_fields' => $fields ) );
									}
									echo ( true == $has_tabs ) ? '</div>' : '';
								endforeach;
								?>
							</div><!--/.mco-panel-content -->
							<?php if ( $has_tabs ): ?>
							</div><!--/.mco-panel-nav-tabs -->
							<?php endif; ?>
						</div>
						<?php } ?>
					</div><!--/.mco-panels -->
				</div><!--/.mco-settings-wrapper -->
				<?php

			}

			if ( $is_to ) {
				echo '</form></div>';
			}

		}

		/*
		 * Render form fields
		 */
		public function render_fields( $fields, $values = false, $display = 'horizontal', $args = array() ) {

			if ( ! is_array( $args ) ) {
				$args = array();
			}

			if ( ! is_array( $values ) ) {
				$values = array();
			}

			if ( ! is_array( $fields ) || ( $display == 'widget' && ! array_key_exists( 'widget', $args ) ) ) {
				return;
			}

			if ( ! array_key_exists( 'all_fields', $args ) ) {
				$args['all_fields'] = $fields;
			}

			$field_types = array();
			foreach ( $fields as $k => $v ) {
				if ( is_array( $v ) && array_key_exists( 'type', $v ) ) {
					$field_types[] = $v['type'];
				}
			}
			$field_types = array_unique( $field_types );

			if ( in_array( 'sidebar_select', $field_types ) ) {

				/* get registered sidebars */
				$wp_registered_sidebars = $this->get_registered_sidebars();

				$sidebar_choices = array();
				if ( is_array( $wp_registered_sidebars ) ) {
					foreach ( $wp_registered_sidebars as $i => $s ) {
						$sidebar_choices[] = array(
							'label' => $s,
							'value' => $i
						);
					}
				}

			}

			if ( in_array( 'taxonomy_checker', $field_types ) || in_array( 'post_type_checker', $field_types ) ) {

				/* get registered post types */
				$post_types = get_post_types( '', false );

				/* these post types will not be retrieved */
				$forbidden_post_types = array( 'attachment', 'revision', 'nav_menu_item' );

				foreach ( $post_types as $k => $v ) {
					if ( in_array( $k, $forbidden_post_types ) || ! $v->public ) {
						unset( $post_types[ $k ] );
						continue;
					}
					$post_types[ $k ] = array(
						'label' => $v->labels->name,
						'value' => $k
					);
				}

				$taxonomies = get_taxonomies( '', false );

				$taxonomy_objects = array();
				foreach ( $post_types as $k => $v ) {
					$object_taxonomies = get_object_taxonomies( $k );
					foreach ( $object_taxonomies as $t ) {
						if (  ! array_key_exists( $t, $taxonomy_objects ) ) {
							$taxonomy_objects[ $t ] = array();
						}
						$taxonomy_objects[ $t ][] = $v;
					}
				}

				$term_taxonomies = array_filter( array_keys( $taxonomy_objects ), function( $v ) {
					return ( ! in_array( $v, array( 'post_format', 'post_tag', 'product_tag' ) ) );
				} );

				$terms = get_terms( $term_taxonomies, array(
					'hide_empty' => 0,
					'orderby' => 'term_group'
				) );

				$taxonomy_choices = array();
				foreach ( $terms as $k => $v ) {
					if ( $taxonomies[ $v->taxonomy ]->public ) {
						$taxonomy_choices[] = array(
							'label' => $v->name .'<small>'. $taxonomies[ $v->taxonomy ]->labels->singular_name .'</small>',
							'value' => $v->term_id .'_'. $v->taxonomy
						);
					}
				}

			}

			if ( in_array( 'advertisement', $field_types ) && class_exists( 'Agg_Flex_Ads' ) ) {

				$ads_obj = Agg_Flex_Ads::get_instance();
				$ads_by_locations = agg_flex_ads_get_ads();
				$ads_payload = $ad_locations = array();

				foreach ( $ads_obj->ad_locations as $location ) {

					$ad_locations[ $location['location'] ] = $location['title'];

					if ( array_key_exists( 'parameters', $location ) ) {
						foreach ( $location['parameters'] as $location_param ) {
							if ( $location_param['id'] == 'position' ) {

								foreach( $location_param['choices'] as $location_position ) {
									$ad_locations[ $location['location'] .'_'. $location_position['value'] ] = $location_position['text'];
								}

								unset( $ad_locations[ $location['location'] ] );
								break;

							}
						}
					}

				}

				foreach ( $ads_by_locations as $ad_location ) {

					foreach ( $ad_location['ads'] as $ad_location_ad ) {

						$ad_location_ad_options = get_post_meta( $ad_location_ad['id'], Agg_Flex_Ads_Ad::$options_meta_field, true );

						$ad_location_id = $ad_location['id'];
						if ( is_array( $ad_location_ad_options ) && array_key_exists( 'position', $ad_location_ad_options ) ) {
							$ad_location_id .= '_'. $ad_location_ad_options['position'];
						}

						if ( ! array_key_exists( $ad_location_id, $ad_locations ) ) {
							continue;
						}

						if ( ! array_key_exists( $ad_location_id, $ads_payload ) ) {

							$ads_payload[ $ad_location_id ] = array(
								'id' => $ad_location_id,
								'title' => $ad_locations[ $ad_location_id ],
								'ads' => array()
							);

						}

						$ads_payload[ $ad_location_id ]['ads'][] = $ad_location_ad;

					}

				}

			}

			if ( ! in_array( $display, array( 'term_edit', 'term_add', 'inline' ) ) ) {
				echo '<div class="mco-form-wrapper">';
			}

			foreach ( $fields as $v ) {

				if ( ! is_array( $v ) || ! array_key_exists( 'id', $v ) || ! array_key_exists( 'type', $v ) ) {
					continue;
				}

				if ( $v['type'] == 'taxonomy_checker' ) {
					$v['type'] = 'checker';
					/** @noinspection PhpUndefinedVariableInspection */
					$v['choices'] = $taxonomy_choices;
				}

				if ( $v['type'] == 'post_type_checker' ) {
					$v['type'] = 'checker';
					/** @noinspection PhpUndefinedVariableInspection */
					$v['choices'] = $post_types;
				}

				if ( $v['type'] == 'sidebar_select' ) {
					$v['type'] = 'select';
					/** @noinspection PhpUndefinedVariableInspection */
					$v['choices'] = $sidebar_choices;
				}

				$form_id = ( $display == 'widget' ) ? $args['widget']->get_field_id( $v['id'] ) : ( ( strpos( $v['id'], 'mco-' ) !== 0 ? 'mco-' : '' ) . $v['id'] );
				$form_data_id = array_key_exists( 'data_id', $v ) ? $v['data_id'] : false;
				$form_name = ( $display == 'widget' ) ? $args['widget']->get_field_name( $v['id'] ) : $v['id'];
				$form_std = array_key_exists( 'std', $v ) ? $v['std'] : '';
				$form_value = array_key_exists( $v['id'], $values ) ? $values[ $v['id'] ] : $form_std;

				if ( array_key_exists( 'name', $v ) ) {
					$form_name = $v['name'];
				}

				$mco_form_wrap_tag = $mco_form_inner_wrap_tag = 'div';
				if ( $display == 'term_edit' ) {
					$mco_form_wrap_tag = 'tr';
					$mco_form_inner_wrap_tag = 'th';
				}

				$mco_form_classes = array(
					'mco-form-block',
					'mco-form-'. str_replace( '_', '-', $v['type'] )
				);

				if ( in_array( $display, array( 'term_edit', 'term_add' ) ) ) {
					$mco_form_classes[] = 'form-field';
					$mco_form_classes[] = str_replace( '_', '-', $form_id ) .'-wrap';
				}

				if ( array_key_exists( 'class', $v ) ) {
					$v['class'] = preg_replace( '#\s+#', ' ', $v['class'] );
					$mco_form_classes = array_merge( $mco_form_classes, explode( ' ', $v['class'] ) );
				}

				if ( ! array_key_exists( 'operator', $v ) || ! in_array( $v['operator'], array( 'and', 'or' ) ) ) {
					$v['operator'] = 'and';
				}

				if ( ! array_key_exists( 'condition', $v ) || ! is_string( $v['condition'] ) ) {
					$v['condition'] = '';
				}

				if ( ! $this->field_is_visible( $v['condition'], $v['operator'], $args['all_fields'], $values ) ) {
					$mco_form_classes[] = 'hide';
				}

				$mco_cond_attr = empty( $v['condition'] ) ? '' : ' data-condition="'. esc_attr( $v['condition'] ) .'"';
				$mco_operator_attr = empty( $mco_cond_attr ) ? '' : ' data-operator="'. esc_attr( $v['operator'] ) .'"';

				if ( $display != 'inline' ) {
					echo '<'. $mco_form_wrap_tag .' class="'. esc_attr( implode( ' ', $mco_form_classes ) ) .'" data-type="'. esc_attr( $v['type'] ) .'" data-id="'. esc_attr( $v['id'] ) .'"'. $mco_cond_attr . $mco_operator_attr .'><'. $mco_form_inner_wrap_tag .' class="mco-inner">';
				}

				echo '<label for="'. esc_attr( $form_id ) .'">'. esc_html( $v['label'] ) .'</label>';

				if ( $display == 'term_edit' ) {
					echo '</th><td>';
				}

				if ( $v['type'] == 'select' && array_key_exists( 'choices', $v ) && is_array( $v['choices'] ) ) {

					$v['choices'] = array_values( $v['choices'] );

					echo '<select id="'. esc_attr( $form_id ) .'" name="'. esc_attr( $form_name ) .'" class="mco-form-control"'. ( $form_data_id ? ' data-id="'. esc_attr( $form_data_id ) .'"' : '' ) .'>';
					foreach ( $v['choices'] as $j ) {

						$has_optgroup = ( array_key_exists( 'type', $j ) && $j['type'] == 'optgroup' );

						if ( $has_optgroup ) {
							echo '<optgroup label="' . esc_attr( $j['label'] ) .'">';
 						} else {
 							$j['value'] = array(
 								array(
	                                'label' => $j['label'],
							        'value' => $j['value']
							    )
						    );
						}

						foreach ( $j['value'] as $v ) {
							echo '<option value="'. esc_attr( $v['value'] ) .'"'. ( $form_value == $v['value'] ? ' selected' : '' ) .'>'. esc_html( $v['label'] ) .'</option>';
						}

						if ( $has_optgroup ) {
							echo '</optgroup>';
						}

					}
					echo '</select>';

				} else if ( $v['type'] == 'text' ) {

					echo '<input type="'. ( array_key_exists( 'input_type', $v ) ? $v['input_type'] : 'text' ) .'" id="'. esc_attr( $form_id ) .'" name="'. esc_attr( $form_name ) .'" value="'. esc_attr( wp_unslash( $form_value ) ) .'" class="mco-form-control" autocomplete="off"'. ( $form_data_id ? ' data-id="'. esc_attr( $form_data_id ) .'"' : '' ) .' />';

				} else if ( $v['type'] == 'textarea' ) {

					echo '<textarea id="'. esc_attr( $form_id ) .'" name="'. esc_attr( $form_name ) .'" class="mco-form-control"'. ( $form_data_id ? ' data-id="'. esc_attr( $form_data_id ) .'"' : '' ) .'>'. esc_textarea( wp_unslash( $form_value ) ) .'</textarea>';

				} else if ( $v['type'] == 'colorpicker' ) {

					$is_alpha = ( in_array( 'alpha', $v ) && $v['alpha'] ) ? 'true' : 'false';

					$labels = array(
						array(
							'id'    => $form_id,
							'name'  => $form_name,
							'value' => is_array( $form_value ) ? '' : $form_value,
							'std'   => is_array( $form_std ) ? '' : $form_std,
							'label' => false,
							'alpha' => $is_alpha
						)
					);

					if ( $form_data_id ) {
						$labels[0]['data_id'] = $form_data_id;
					}

					if ( array_key_exists( 'labels', $v ) && is_array( $v['labels'] ) ) {

						if ( ! is_array( $form_value ) ) {
							$form_value = array();
						}

						if ( ! is_array( $form_std ) ) {
							$form_std = array();
						}

						$labels = array();

						foreach ( $v['labels'] as $i => $j ) {

							$labels[] = array(
								'id'      => $form_id . '_' . $i,
								'name'    => $form_name . '[' . sanitize_key( $i ) . ']',
								'value'   => array_key_exists( $i, $form_value ) ? $form_value [ $i ] : '',
								'std'     => array_key_exists( $i, $form_std ) ? $form_std[ $i ] : '',
								'label'   => $j,
								'alpha'   => $is_alpha,
								'data_id' => sanitize_key( $i )
							);

						}

					}

					foreach ( $labels as $i => $j ) {

						echo '<div class="mco-wrap">'. ( $j['label'] ? '<label for="'. esc_attr( $j['id'] ) .'">'. esc_html( $j['label'] ) .'</label>' : '' ) .'<input type="text" id="'. esc_attr( $j['id'] ) .'" name="'. esc_attr( $j['name'] ) .'" value="'. esc_attr( $j['value'] ) .'" data-default-color="'. esc_attr( $j['std'] ) .'" class="mco-form-control" autocomplete="off" data-alpha="'. esc_attr( $is_alpha ) .'"'. ( array_key_exists( 'data_id', $j ) ? ' data-id="'. $j['data_id'] .'"' : '' ) . ' /></div>';

					}

				} else if ( $v['type'] == 'range' ) {

					$slider_atts = '';
					if ( array_key_exists( 'minmax', $v ) && preg_match( '#^([0-9]+)\,([0-9]+)$#', $v['minmax'], $match ) ) {
						$slider_atts = ' data-min="'. intval( $match[1] ) .'" data-max="'. intval( $match[2] ) .'"';
					}

					$unit = ( array_key_exists( 'unit', $v ) && is_string( $v['unit'] ) ) ? $v['unit'] : '';

					if ( ! is_numeric( $form_value ) ) {
						$form_value = empty( $match ) ?  0 : intval( $match[1] );
					}

					echo '<div class="mco-wrap">
						<div class="mco-slider" data-value="'. intval( $form_value ) .'"'. $slider_atts .'></div>
					  	<div class="mco-input-wrap">
					  		<input type="text" id="'. esc_attr( $form_id ) .'" name="'. esc_attr( $form_name ) .'" class="mco-form-control" placeholder="'. intval( $form_value ) .'" value="'. intval( $form_value ) .'" autocomplete="off"'. ( $form_data_id ? ' data-id="'. esc_attr( $form_data_id ) .'"' : '' ) .' />
							<span class="mco-unit">'. esc_html( $unit ) .'</span>
					  	</div>
					  </div>';

				} else if ( $v['type'] == 'toggle' ) {

					$toggle_opt = ( $form_value == 'true' ) ? array( 'on', 'true' ) : array( 'off', 'false' );

					echo '<div id="'. esc_attr( $form_id ) .'" class="mco-wrap '. esc_attr( $toggle_opt[0] ) .'">
							<span class="mco-circle"></span>
							<span class="mco-label-on">'. esc_html__( 'ON', 'sanfrancisco' ) .'</span>
							<span class="mco-label-off">'. esc_html__( 'OFF', 'sanfrancisco' ) .'</span>
							<input type="hidden" name="'. esc_attr( $form_name ) .'" class="mco-form-control" value="'. esc_attr( $toggle_opt[1] ) .'"'. ( $form_data_id ? ' data-id="'. esc_attr( $form_data_id ) .'"' : '' ) .' />
						</div>';

				} else if ( $v['type'] == 'media' ) {

					if ( ! array_key_exists( 'media_type', $v ) ) {
						$v['media_type'] = 'image';
					}

					$mco_attachment = array(
						'url'     => '',
						'preview' => false
					);

					if ( is_numeric( $form_value ) ) {

						if ( $v['media_type'] == 'image' ) {
							$mco_attachment['url'] = wp_get_attachment_image_url( $form_value, 'full' );
							$mco_attachment['preview'] = wp_get_attachment_image( $form_value, 'full' );
						} else if ( $v['media_type'] == 'video' ) {
							$mco_attachment['url'] = wp_get_attachment_url( $form_value );
						} else if ( $v['media_type'] == 'audio' ) {
							$mco_attachment['url'] = wp_get_attachment_url( $form_value );
						}

					}

					echo '<div class="mco-wrap" data-type="'. $v['media_type'] .'">
						<input type="text" id="'. esc_attr( $form_id ) .'" value="'. esc_url( $mco_attachment['url'] ) .'" class="mco-url" autocomplete="off" readonly="true" />
						<input type="hidden" name="'. esc_attr( $form_name ) .'" class="mco-form-control" value="'. esc_attr( $form_value ) .'"'. ( $form_data_id ? ' data-id="'. esc_attr( $form_data_id ) .'"' : '' ) .' />
						<button type="button" class="mco-btn mco-btn-pink mco-btn-clear">'. esc_html__( 'Clear', 'sanfrancisco' ) .'</button>
						<button type="button" class="mco-btn mco-btn-darkblue mco-btn-upload">'. esc_html__( 'Upload', 'sanfrancisco' ) .'</button>
					</div>';

					if ( $mco_attachment['preview'] && ( ! array_key_exists( 'preview', $v ) || $v['preview'] ) ) {
						echo '<div class="mco-preview">'. $mco_attachment['preview'] .'</div>';
					}

				} else if ( $v['type'] == 'gallery' ) {

					if ( ! is_array( $form_value ) ) {
						$form_value = array();
					}

					?>
					<div class="mco-wrap" data-name="<?php echo esc_attr( $form_name ); ?>">
						<?php if ( ! empty( $form_value ) ): ?>
						<ul class="mco-preview">
							<?php foreach ( $form_value as $i => $j ): if ( ! ( $attachment_image = wp_get_attachment_image_url( $j, 'thumbnail' ) ) ) continue; ?>
							<li class="mco-gallery-image">
								<span class="mco-btn-remove"><i class="fa fa-times"></i></span>
								<img src="<?php echo esc_url( $attachment_image ); ?>" />
								<input type="hidden" name="<?php echo esc_attr( $form_name ) ?>[<?php echo intval( $i ); ?>]" class="mco-form-control" value="<?php echo intval( $j ); ?>" data-id="<?php echo intval( $i ); ?>" />
							</li>
							<?php endforeach; ?>
						</ul>
						<?php endif; ?>
						<button type="button" class="mco-btn mco-btn-green mco-btn-upload"><?php echo esc_html__( 'Add Images', 'sanfrancisco' ); ?></button>
					</div>
					<?php

				} else if ( $v['type'] == 'checker' && array_key_exists( 'choices', $v ) && is_array( $v['choices'] ) ) {

					if ( $form_value != 'all' ) {

						if ( is_string( $form_value ) ) {
							$form_value = array_filter( explode( ',', $form_value ), function( $mv ) { return trim( $mv ); } );
						}

						if ( ! is_array( $form_value ) ) {
							$form_value = array();
						}

					}

					$v['choices'] = array_values( $v['choices'] );

					echo '<div id="'. esc_attr( $form_id ) .'" class="mco-wrap"><ul>';

					foreach ( $v['choices'] as $i => $j ) {

						$choice_class = array( 'mco-choice' );

						if ( ! array_key_exists( 'operator', $j ) || ! in_array( $j['operator'], array( 'and', 'or' ) ) ) {
							$j['operator'] = 'and';
						}

						if ( ! array_key_exists( 'condition', $j ) || ! is_string( $j['condition'] ) ) {
							$j['condition'] = '';
						}

						if ( ! $this->field_is_visible( $j['condition'], $j['operator'], $args['all_fields'], $values ) ) {
							$choice_class[] = 'hide';
						}

						$cond_attr = empty( $j['condition'] ) ? '' : ' data-condition="'. esc_attr( $j['condition'] ) .'"';
						$operator_attr = empty( $mco_cond_attr ) ? '' : ' data-operator="'. esc_attr( $j['operator'] ) .'"';

						echo '<li class="' . esc_attr( implode( ' ', $choice_class ) ) . '"' . $cond_attr . $operator_attr . '><label><input type="checkbox" name="'. esc_attr( $form_name ) .'['. intval( $i ) .']" class="mco-form-control" value="'. esc_attr( $j['value'] ) .'" data-id="'. intval( $i ) . '" ' . ( ( $form_value == 'all' || in_array( $j['value'], $form_value ) ) ? ' checked' : '' ) . '/> ' . wp_kses( $j['label'], array( 'small' => array() ) ) .'</label></li>';

					}

					echo '</ul></div>';

				} else if ( $v['type'] == 'radio_select' ) {

					echo '<div id="'. esc_attr( $form_id ) .'" class="mco-wrap"><ul>';

					foreach ( $v['choices'] as $j ) {

						$choice_value = '';
						$choice_class = array();

						if ( array_key_exists( 'text', $j ) ) {
							$choice_value = esc_html( $j['text'] );
							$choice_class[] = 'mco-choice-text';
						} else if ( array_key_exists( 'image', $j ) ) {
							$choice_value = '<img src="'. esc_url( $j['image'] ) .'"'. ( array_key_exists( 'alt', $j ) ? ' alt="'. esc_attr( $j['alt'] ) .'"' : '' ) .' />';
							$choice_class[] = 'mco-choice-image';
						} else if ( array_key_exists( 'html', $j ) ) {
							$choice_value = wp_filter_post_kses( $j['html'] );
							$choice_class[] = 'mco-choice-html';
						}

						if ( $j['value'] == $form_value ) {
							$choice_class[] = 'active';
						}

						if ( ! array_key_exists( 'operator', $j ) || ! in_array( $j['operator'], array( 'and', 'or' ) ) ) {
							$j['operator'] = 'and';
						}

						if ( ! array_key_exists( 'condition', $j ) || ! is_string( $j['condition'] ) ) {
							$j['condition'] = '';
						}

						if ( ! $this->field_is_visible( $j['condition'], $j['operator'], $args['all_fields'], $values ) ) {
							$choice_class[] = 'hide';
						}

						$cond_attr = empty( $j['condition'] ) ? '' : ' data-condition="'. esc_attr( $j['condition'] ) .'"';
						$operator_attr = empty( $mco_cond_attr ) ? '' : ' data-operator="'. esc_attr( $j['operator'] ) .'"';

						echo '<li class="'. esc_attr( implode( ' ', $choice_class ) ) .'" data-value="'. esc_attr( $j['value'] ) .'"' . $cond_attr . $operator_attr . '>'. $choice_value .'</li>';

					}

					echo '</ul><input type="hidden" name="'. esc_attr( $form_name ) .'" class="mco-form-control" value="'. esc_attr( $form_value ) .'" ' . ( $form_data_id ? 'data-id="'. esc_attr( $form_data_id ) .'" ' : '' ) . '/></div>';

				} else if ( $v['type'] == 'sidebar_create' ) {

					if ( ! is_array( $form_value ) ) {
						$form_value = array();
					}

					$placeholder_text = esc_html__( 'Type sidebar name...', 'sanfrancisco' );

					?>
					<div class="mco-wrap" data-name="<?php echo esc_attr( $form_name ); ?>" data-max-id="<?php echo count( $form_value ) ? max( array_keys( $form_value ) ) : 0; ?>" data-placeholder="<?php echo esc_attr( $placeholder_text ); ?>">
						<?php foreach ( $form_value as $i => $j ): ?>
						<div class="mco-item-block">
							<input type="text" name="<?php echo esc_attr( $form_name ); ?>[<?php echo intval( $i ); ?>]" value="<?php echo esc_attr( wp_unslash( $j ) ); ?>" data-id="<?php echo intval( $i ); ?>" class="mco-form-control" placeholder="<?php echo esc_attr( $placeholder_text ); ?>" autocomplete="off" />
							<button type="button" class="mco-btn mco-btn-red mco-btn-remove"><span class="fa fa-trash"></span></button>
						</div>
						<?php endforeach; ?>
						<button type="button" class="mco-btn mco-btn-green mco-btn-new"><span class="fa fa-plus"></span> <?php esc_html_e( 'Add New', 'sanfrancisco' ); ?></button>
					</div>
					<?php

				} else if ( $v['type'] == 'code' ) {

					if ( ! array_key_exists( 'editor', $v ) ) {
						$v['editor'] = 'html';
					}

					$editor_theme = ( $v['editor'] == 'css' ) ? 'monokai' : 'chrome';

					echo '<textarea class="mco-form-control" name="'. esc_attr( $form_name ) .'" style="display: none;"'. ( $form_data_id ? ' data-id="'. esc_attr( $form_data_id ) .'"' : '' ) .'>'. htmlspecialchars( wp_unslash( $form_value ) ) .'</textarea>';
					echo '<div id="'. esc_attr( $form_id ) .'" class="mco-code-editor" data-name="'. esc_attr( $form_name ) .'" data-mode="'. esc_attr( $v['editor'] ) .'" data-theme="'. esc_attr( $editor_theme ) .'">'. htmlspecialchars( wp_unslash( $form_value ) ) .'</div>';

				} else if ( $v['type'] == 'wp_editor' ) {

					if ( ! array_key_exists( 'args', $v ) ) {
						$v['args'] = array();
					}

					$form_id .= '_'. uniqid();

					wp_editor( wp_unslash( $form_value ), $form_id, wp_parse_args( $v['args'], array(
						'textarea_name'    => $form_name,
						'textarea_rows'    => 20,
						'editor_class'     => 'mco-form-control',
						'drag_drop_upload' => true
					) ) );

				} else if ( $v['type'] == 'font' ) {

					if ( ! is_array( $form_value ) ) {
						$form_value = array();
					}

					if ( ! is_array( $form_std ) ) {
						$form_std = array();
					}

					$font_fields = array(
						'font_family'        => array(
							'label' => esc_html__( 'Font Family', 'sanfrancisco' ),
							'type'  => 'select2'
						),
						'backup_font_family' => array(
							'label' => esc_html__( 'Backup Font Family', 'sanfrancisco' ),
							'type'  => 'select2'
						),
						'font_weight_style'  => array(
							'label' => esc_html__( 'Font Weight & Style', 'sanfrancisco' ),
							'type'  => 'select2'
						),
						'font_subsets'       => array(
							'label' => esc_html__( 'Font Subsets', 'sanfrancisco' ),
							'type'  => 'select2'
						),
						'font_size'          => array(
							'label'  => esc_html__( 'Font Size', 'sanfrancisco' ),
							'type'   => 'range',
							'minmax' => '10,72',
							'unit'   => 'px',
						)
					);

					if ( ! array_key_exists( 'properties', $v ) || ! is_array( $v['properties'] ) || empty( $v['properties'] ) ) {
						$v['properties'] = array_keys( $font_fields );
					}

					$field_output = array();
					foreach ( $font_fields as $i => $j ) {

						$j = wp_parse_args( array(
							'id'      => sanitize_key( $form_id ) . '_' . $i,
							'data_id' => $i,
							'std'     => array_key_exists( $i, $form_std ) ? $form_std[ $i ] : '',
							'name'    => $form_name . '[' . $i . ']'
						), $j );

						$field_render_value = array();
						$field_render_value[ $j['id'] ] = array_key_exists( $i, $form_value ) ? $form_value[ $i ] : '';

						if ( $j['type'] == 'select2' ) {

							$field_output[ $i ] = '<label for="'. esc_attr( $j['id'] ) .'">'. $j['label'] .'</label>';
							$field_output[ $i ] .= '<select id="'. esc_attr( $j['id'] ) .'" class="mco-form-control" name="'. esc_attr( $j['name'] ) .'" data-id="'. esc_attr( $i ) .'" data-placeholder="'. esc_attr( $j['label'] ) .'" data-allow-clear="true" data-value="'. esc_attr( wp_unslash( $field_render_value[ $j['id'] ] ) ) .'"></select>';

						} else {

							ob_start();
							$this->render_fields( array( $j ), $field_render_value, 'inline', $args );
							$field_output[ $i ] = ob_get_clean();

						}

					}

					echo '<div class="mco-group-wrap">';
					foreach ( $field_output as $i => $j ) {
						if ( in_array( $i, $v['properties'] ) ) {
							echo '<div class="mco-group-field mco-form-'. sanitize_html_class( $font_fields[ $i ]['type'] ) .' mco-typo-'. sanitize_html_class( str_replace( '_', '-', $i ) ) .'">'. $j .'</div>';
						}
					}
					echo '</div>';

				} else if ( $v['type'] == 'background' ) {

					if ( ! is_array( $form_value ) ) {
						$form_value = array();
					}

					if ( ! is_array( $form_std ) ) {
						$form_std = array();
					}

					$bg_fields = array(
						'background_image'      => array(
							'label' => esc_html__( 'Background Image', 'sanfrancisco' ),
							'type' => 'media',
							'media_type' => 'image'
						),
						'background_repeat'     => array(
							'label' => esc_html__( 'Background Repeat', 'sanfrancisco' ),
							'type' => 'select',
							'choices' => array(
								array(
									'label' => esc_html__( 'No Repeat', 'sanfrancisco' ),
									'value' => 'no_repeat'
								),
								array(
									'label' => esc_html__( 'Tile', 'sanfrancisco' ),
									'value' => 'repeat'
								),
								array(
									'label' => esc_html__( 'Tile Horizontally', 'sanfrancisco' ),
									'value' => 'repeat_x'
								),
								array(
									'label' => esc_html__( 'Tile Vertically', 'sanfrancisco' ),
									'value' => 'repeat_y'
								)
							)
						),
						'background_position'   => array(
							'label' => esc_html__( 'Background Position', 'sanfrancisco' ),
							'type' => 'select',
							'choices' => array(
								array(
									'label' => esc_html__( 'Top Left', 'sanfrancisco' ),
									'value' => 'top_left'
								),
								array(
									'label' => esc_html__( 'Top Center', 'sanfrancisco' ),
									'value' => 'top_center'
								),
								array(
									'label' => esc_html__( 'Top Right', 'sanfrancisco' ),
									'value' => 'top_right'
								),
								array(
									'label' => esc_html__( 'Center Left', 'sanfrancisco' ),
									'value' => 'center_left'
								),
								array(
									'label' => esc_html__( 'Center Center', 'sanfrancisco' ),
									'value' => 'center_center'
								),
								array(
									'label' => esc_html__( 'Center Right', 'sanfrancisco' ),
									'value' => 'center_right'
								),
								array(
									'label' => esc_html__( 'Bottom Left', 'sanfrancisco' ),
									'value' => 'bottom_left'
								),
								array(
									'label' => esc_html__( 'Bottom Center', 'sanfrancisco' ),
									'value' => 'bottom_center'
								),
								array(
									'label' => esc_html__( 'Bottom Right', 'sanfrancisco' ),
									'value' => 'bottom_right'
								)
							)
						),
						'background_attachment' => array(
							'label' => esc_html__( 'Background Attachment', 'sanfrancisco' ),
							'type' => 'select',
							'choices' => array(
								array(
									'label' => esc_html__( 'Scroll', 'sanfrancisco' ),
									'value' => 'scroll'
								),
								array(
									'label' => esc_html__( 'Fixed', 'sanfrancisco' ),
									'value' => 'fixed'
								)
							)
						),
						'background_size'       => array(
							'label' => esc_html__( 'Background Size', 'sanfrancisco' ),
							'type' => 'text'
						),
						'background_color'      => array(
							'label' => esc_html__( 'Background Color', 'sanfrancisco' ),
							'type' => 'colorpicker',
							'alpha' => true
						)
					);

					if ( ! array_key_exists( 'properties', $v ) || ! is_array( $v['properties'] ) || empty( $v['properties'] ) ) {
						$v['properties'] = array_keys( $bg_fields );
					}

					$field_output = array();
					foreach ( $bg_fields as $i => $j ) {

						$j = wp_parse_args( array(
							'id'      => sanitize_key( $form_id ) . '_' . $i,
							'data_id' => $i,
							'std'     => array_key_exists( $i, $form_std ) ? $form_std[ $i ] : '',
							'name'    => $form_name . '[' . $i . ']'
						), $j );

						$field_render_value = array();
						$field_render_value[ $j['id'] ] = array_key_exists( $i, $form_value ) ? $form_value[ $i ] : '';

						ob_start();
						$this->render_fields( array( $j ), $field_render_value, 'inline', $args );
						$field_output[ $i ] = ob_get_clean();

					}

					echo '<div class="mco-group-wrap">';
					foreach ( $field_output as $i => $j ) {
						if ( in_array( $i, $v['properties'] ) ) {
							echo '<div class="mco-group-field mco-form-'. sanitize_html_class( $bg_fields[ $i ]['type'] ) . ( $i == 'background_image' ? ' mco-group-field-wide' : '' ) .'">'. $j .'</div>';
						}
					}
					echo '</div>';

				} else if ( $v['type'] == 'advertisement' && isset( $ads_payload ) ) {

					if ( ! is_array( $form_value ) ) {
						$form_value = array();
					}

					if ( ! array_key_exists( 'locations', $v ) || ! is_array( $v['locations'] ) || empty( $v['locations'] ) ) {
						$v['locations'] = array_keys( $ads_payload );
					}

					$ad_device_preferences = array(
						'lg' => esc_html__( 'Desktop (Large)', 'sanfrancisco' ),
						'md' => esc_html__( 'Desktop (Small)', 'sanfrancisco' ),
						'sm' => esc_html__( 'Tablet', 'sanfrancisco' ),
						'xs' => esc_html__( 'Mobile', 'sanfrancisco' )
					);

					$field_output = array();
					foreach ( $ads_payload as $ad_location ) {

						foreach ( $ad_device_preferences as $breakpoint_id => $dp_title ) {

							$j = array(
								'id'      => sanitize_key( $form_id ) . '_' . $ad_location['id'] .'_'. $breakpoint_id,
								'label'   => $ad_location['title'],
								'data_id' => $ad_location['id'] .'_'. $breakpoint_id,
								'type'    => 'select',
								'std'     => 'none',
								'name'    => $form_name . '[' . $ad_location['id'] . ']['. $breakpoint_id .']',
								'choices' => array(
									array(
										'label' => esc_html__( 'Disabled', 'sanfrancisco' ),
										'value' => 'none'
									)
								)
							);

							foreach ( $ad_location['ads'] as $ad_location_ad ) {
								$j['choices'][] = array(
									'label' => $ad_location_ad['title'],
									'value' => $ad_location_ad['id']
								);
							}

							$field_render_value = array();
							$field_render_value[ $j['id'] ] = ( array_key_exists( $ad_location['id'], $form_value ) && is_array( $form_value[ $ad_location['id'] ] ) && array_key_exists( $breakpoint_id, $form_value[ $ad_location['id'] ] ) ) ? $form_value[ $ad_location['id'] ][ $breakpoint_id ] : 'none';

							ob_start();
							$this->render_fields( array( $j ), $field_render_value, 'inline', $args );
							$field_output[ $breakpoint_id ][ $ad_location['id'] ] = ob_get_clean();

						}

					}

					echo '<div class="mco-form-block-nav-tabs"><ul role="tablist">';
					foreach ( $ad_device_preferences as $breakpoint_id => $breakpoint_name ) {
						/** @noinspection HtmlUnknownAnchorTarget */
						echo '<li role="presentation"'. ( $breakpoint_id == 'lg' ? ' class="active"' : '' ) .'><a href="#'. $form_name .'-'. $breakpoint_id .'" aria-controls="'. $form_name .'-'. $breakpoint_id .'" role="tab" data-toggle="tab" aria-expanded="true">'. esc_html( $breakpoint_name ) .'</a></li>';
					}
					echo '</ul></div><!-- .mco-form-block-nav-tabs -->';

					echo '<div class="mco-group-wrap mco-form-block-panels">';
					foreach ( $field_output as $breakpoint_id => $o ) {
						echo '<div role="tabpanel" class="mco-form-block-panel fade'. ( $breakpoint_id == 'lg' ? ' active in' : '' ) .'" id="'. $form_name .'-'. $breakpoint_id .'">';
						foreach ( $o as $i => $j ) {
							if ( in_array( $i, $v['locations'] ) ) {
								echo '<div class="mco-group-field mco-form-select">' . $j . '</div>';
							}
						}
						echo '</div><!-- .mco-form-block-panel -->';
					}
					echo '</div><!-- .mco-group-wrap -->';

				}

				if ( array_key_exists( 'desc', $v ) && ! empty( $v['desc'] ) ) {
					echo '<p class="mco-description">'. esc_html( $v['desc'] ) .'</p>';
				}

				if ( $display != 'inline' ) {

					if ( $display != 'term_edit' ) {
						echo '</'. $mco_form_inner_wrap_tag .'><!-- .mco-inner -->';
					} else {
						echo '</td>';
					}

					echo '</'. $mco_form_wrap_tag .'><!-- .mco-form-block -->';

				}

			}

			if ( ! in_array( $display, array( 'term_edit', 'term_add', 'inline' ) ) ) {
				echo '</div><!-- form-wrapper -->';
			}

		}

		/*
		 * Sanitize form fields
		 */
		public static function sanitize_fields( $values, $fields ) {

			$result = array();

			if ( ! is_array( $values ) || ! is_array( $fields ) ) {
				return $result;
			}

			foreach ( $fields as $field ) {

				if ( ! array_key_exists( 'id', $field )
					|| ! array_key_exists( 'type', $field )
					/*|| ( defined( 'DOING_AJAX' ) && DOING_AJAX && ! array_key_exists( $field['id'], $values ) ) */) {
					continue;
				}

				$id = $field['id'];
				$hex_pattern = '#^(?:(rgba?)\(\s*([0-9]{1,3})\s*\,\s*([0-9]{1,3})\s*\,\s*([0-9]{1,3})\s*(?:\,\s*((?:0\.*(?:[0-9]{1,2})*)|1)\s*)*\)|(\#[a-f0-9]{6}))$#i';

				if ( in_array( $field['type'], array( 'text', 'textarea' ) ) ) {

					if ( array_key_exists( $id, $values ) ) {
						$value = sanitize_text_field( $values[ $id ] );
					}

					if ( ! isset( $value ) ) {
						$value = ( array_key_exists( 'std', $field ) ) ? sanitize_text_field( $field['std'] ) : '';
					}

					$value = wp_unslash( $value );

				} else if ( $field['type'] == 'range' ) {

					if ( array_key_exists( $id, $values ) && array_key_exists( 'minmax', $field ) ) {

						$values[ $id ] = intval( $values[ $id ] );

						if ( is_string( $field['minmax'] ) ) {
							$field['minmax'] = explode( ',', $field['minmax'] );
						}

						if ( is_array( $field['minmax'] ) && count( $field['minmax'] ) == 2 ) {

							list( $min, $max ) = array_map( function ( $v ) {
								return intval( trim( $v ) );
							}, $field['minmax'] );

							if ( $values[ $id ] >= $min && $values[ $id ] <= $max ) {
								$value = $values[ $id ];
							}

						}

					}

					if ( ! isset( $value ) ) {
						$value = ( array_key_exists( 'std', $field ) ) ? $field['std'] : 0;
					}

					$value = intval( $value );

				} else if ( in_array( $field['type'], array( 'checker', 'taxonomy_checker', 'post_type_checker', 'gallery' ) ) ) {

					if ( array_key_exists( $id, $values ) ) {

						if ( is_string( $values[ $id ] ) ) {
							$values[ $id ] = explode( ',', $values[ $id ] );
						}

						if ( is_array( $values[ $id ] ) ) {
							$value = $values[ $id ];
						}

					}

					if ( ! isset( $value ) ) {
						$value = array();
					}

				} else if ( in_array( $field['type'], array( 'select', 'sidebar_select', 'radio_select' ) ) ) {

					if ( array_key_exists( $id, $values ) ) {
						$value = sanitize_key( $values[ $id ] );
					} else {
						$value = ( array_key_exists( 'std', $field ) ) ? sanitize_key( $field['std'] ) : '';
					}

				} else if ( $field['type'] == 'toggle' ) {

					if ( array_key_exists( $id, $values ) && in_array( $values[ $id ], array( 'true', 'false' ), true ) ) {
						$value = $values[ $id ];
					} else {
						$value = ( array_key_exists( 'std', $field ) && in_array( $field['std'], array( 'true', 'false' ), true ) ) ? $field['std'] : 'false';
					}

				} else if ( $field['type'] == 'colorpicker' ) {

					if ( array_key_exists( $id, $values ) ) {

						if ( is_array( $values[ $id ] ) ) {
							$value = array();
							foreach ( $values[ $id ] as $i => $f ) {
								if ( preg_match( $hex_pattern, $f ) ) {
									$hex_val = $f;
								}
								if ( ! isset( $hex_val ) ) {
									$hex_val = ( array_key_exists( 'std', $field ) && is_array( $field['std'] ) && in_array( $i, $field['std'] ) && preg_match( $hex_pattern, $field['std'][ $i ] ) ) ? $field['std'][ $i ] : '';
								}
								$value[ $i ] = $hex_val;
							}
						} else if ( preg_match( $hex_pattern, $values[ $id ] ) ) {
							$value = $values[ $id ];
						} else {
							$value = '';
						}

					} else if ( array_key_exists( 'std', $field ) ) {

						if ( is_array( $field['std'] ) ) {
							$value = array();
							foreach ( $field['std'] as $i => $f ) {
								$value[ $i ] = ( preg_match( $hex_pattern, $f ) ) ? $f : '';
							}
						} else if ( preg_match( $hex_pattern, $field['std'] ) ) {
							$value = $field['std'];
						} else {
							$value = '';
						}

					} else {
						$value = '';
					}

				} else if ( $field['type'] == 'media' ) {

					if ( array_key_exists( $id, $values ) ) {
						$value = is_numeric( $values[ $id ] ) ? intval( $values[ $id ] ) : '';
					} else {
						$value = ( array_key_exists( 'std', $field ) && is_numeric( $field['std'] ) ) ? intval( $field['std'] ) : '';
					}

				} else if ( $field['type'] == 'sidebar_create' ) {

					$value = array();
					if ( array_key_exists( $id, $values ) && is_array( $values[ $id ] ) ) {
						foreach ( $values[ $id ] as $i => $t ) {
							$i = sanitize_key( $i );
							$value[ $i ] = wp_unslash( sanitize_text_field( $t ) );
						}
					}

				} else if ( $field['type'] == 'font' ) {

					$value = array(
						'font_family'        => '',
						'backup_font_family' => '',
						'font_weight_style'  => '',
						'font_subsets'       => '',
						'font_size'          => ''
					);

					if ( array_key_exists( 'properties', $field ) && is_array( $field['properties'] ) ) {
						foreach ( $value as $i => $v ) {
							if ( ! in_array( $i, $field['properties'] ) ) {
								unset( $value[ $i ] );
							}
						}
					}

					foreach ( $value as $i => $v ) {

						$val = ( array_key_exists( $id, $values ) && is_array( $values[ $id ] ) ) ? $values[ $id ] : ( ( array_key_exists( 'std', $field ) && is_array( $field['std'] ) ) ? $field['std'] : false );

						if ( ! $val ) {
							break;
						}

						$value[ $i ] = ( $i == 'font_size' ) ? intval( $val[ $i ] ) : wp_unslash( sanitize_text_field( $val[ $i ] ) );

					}

				} else if ( $field['type'] == 'background' ) {

					$value = array(
						'background_color'      => '',
						'background_image'      => '',
						'background_repeat'     => '',
						'background_position'   => '',
						'background_size'       => '',
						'background_attachment' => ''
					);

					$map = array(
						'background_repeat'     => array(
							'no_repeat',
							'repeat',
							'repeat_x',
							'repeat_y'
						),
						'background_position'   => array(
							'top_left',
							'top_center',
							'top_right',
							'center_left',
							'center_center',
							'center_right',
							'bottom_left',
							'bottom_center',
							'bottom_right'
						),
						'background_attachment' => array(
							'scroll',
							'fixed'
						)
					);

					if ( array_key_exists( 'properties', $field ) && is_array( $field['properties'] ) ) {
						foreach ( $value as $i => $v ) {
							if ( ! in_array( $i, $field['properties'] ) ) {
								unset( $value[ $i ] );
							}
						}
					}

					foreach ( $value as $i => $v ) {

						$val = ( array_key_exists( $id, $values ) && is_array( $values[ $id ] ) ) ? $values[ $id ] : ( ( array_key_exists( 'std', $field ) && is_array( $field['std'] ) ) ? $field['std'] : false );

						if ( ! $val ) {
							break;
						}

						if ( array_key_exists( $i, $val ) ) {

							if ( $i == 'background_color' ) {
								$value[ $i ] = preg_match( $hex_pattern, $val[ $i ] ) ? $val[ $i ] : '';
							} else if ( $i == 'background_image' ) {
								$value[ $i ] = is_numeric( $val[ $i ] ) ? $val[ $i ] : '';
							} else if ( $i == 'background_size' ) {
								$value[ $i ] = sanitize_text_field( $val[ $i ] );
							} else {
								$value[ $i ] = in_array( $val[ $i ], $map[ $i ] ) ? $val[ $i ] : '';
							}
							
						}

					}

				} else if ( $field['type'] == 'code' ) {

					if ( ! array_key_exists( 'editor', $field ) ) {
						$field['editor'] = 'html';
					}

					if ( $field['editor'] == 'html' ) {
						// we need to trust the user here
						$value = array_key_exists( $id, $values ) ? $values[ $id ] : '';
					} else if ( $field['editor'] == 'css' ) {
						$value = array_key_exists( $id, $values ) ? $values[ $id ] : '';
					}

					if ( isset( $value ) ) {
						$value = wp_unslash( $value );
					}

				} else if ( $field['type'] == 'wp_editor' ) {

					if ( array_key_exists( $id, $values ) ) {
						$value = ( $values[ $id ] );
					}

					if ( ! isset( $value ) ) {
						$value = ( array_key_exists( 'std', $field ) ) ? ( $field['std'] ) : '';
					}

					$value = wp_unslash( $value );

				} else if ( $field['type'] == 'advertisement' ) {

					$value = array();
					if ( array_key_exists( $id, $values ) && is_array( $values[ $id ] ) ) {
						foreach ( $values[ $id ] as $k => $v ) {
							foreach ( $v as $i => $j ) {
								if ( is_numeric( $j ) || $j == 'none' ) {
									if ( ! array_key_exists( $k, $value ) ) {
										$value[ $k ] = array();
									}
									$value[ $k ][ $i ] = $j;
								}
							}
						}
					}

				}

				if ( isset( $value ) ) {
					$result[ $id ] = $value;
				}

			}

			return $result;

		}

		/*
		 * Check form conditions
		 */
		public function field_is_visible( $condition, $operator, $fields, $values ) {

			if ( ! is_string( $condition ) || empty( $condition ) ) {
				return true;
			}

			if ( ! is_array( $fields ) ) {
				$fields = array();
			}

			if ( ! is_array( $values ) ) {
				$values = array();
			}

			$field_values = array();
			foreach ( $fields as $v ) {
				$field_values[ $v['id'] ] = array_key_exists( $v['id'], $values ) ? $values[ $v['id'] ] : ( array_key_exists( 'std', $v ) ? $v['std'] : '' );
			}

			$bool_arr = array();
			$cond_arr = array_map( function($v) { $l = substr($v, -1); if ( $l != ')' ) { $v .= ')'; } return $v; }, explode( '),', $condition ) );

			foreach ( $cond_arr as $v ) {

				$bool = false;

				preg_match( '#^([a-z0-9_]+)\:(not|is|has|has_not|in|not_in)\(([a-z0-9_\,]+)\)$#', trim( $v ), $match );

				if ( ! empty( $match ) ) {

					$id = $match[1];
					$op = $match[2];
					$val = $match[3];

					if ( in_array( $op, array( 'is', 'not' ) ) ) {

						$bool = ( array_key_exists( $id, $field_values ) && $field_values[ $id ] == $val );

						if ( $op == 'not' ) {
							$bool = ( ! $bool );
						}

					} else if ( in_array( $op, array( 'has', 'has_not' ) ) ) {

						if ( ! array_key_exists( $id, $field_values ) ) {
							$field_values[ $id ] = array();
						}

						if ( is_string( $field_values[ $id ] ) ) {
							$field_values[ $id ] = array_filter( explode( ',', $field_values[ $id ] ), function( $mv ) { return trim( $mv ); } );
						}

						if ( ! is_array( $field_values[ $id ] ) ) {
							$field_values[ $id ] = array();
						}

						$val = array_filter( explode( ',', $val ), function( $mv ) { return trim( $mv ); } );

						$bool = ( array_intersect( $val, $field_values[ $id ] ) == $val || ( count( $field_values[ $id ] ) == 1 && end( $field_values[ $id ] ) == 'all' ) );

						if ( $op == 'has_not' ) {
							$bool = ( ! $bool );
						}

					} else if ( in_array( $op, array( 'in', 'not_in' ) ) ) {

						$val = array_filter( explode( ',', $val ), function( $mv ) { return trim( $mv ); } );

						$bool = ( array_key_exists( $id, $field_values ) && in_array( $field_values[ $id ], $val ) );

						if ( $op == 'not_in' ) {
							$bool = ( ! $bool );
						}

					}

				}

				$bool_arr[] = $bool;

			}

			if ( $operator == 'or' ) {
				return in_array( true, $bool_arr, true );
			} else {
				return ( ! in_array( false, $bool_arr, true ) );
			}

		}

		/*
		 * Get predefined google & standard web fonts
		 */
		public function get_fonts_map() {

			ob_start();
			include $this->options['lib_path'] .'lib/fonts-map.json';
			$fonts = ob_get_clean();

			if ( ! empty( $fonts ) ) {
				$fonts = json_decode( $fonts, true );
			}

			if ( ! is_array( $fonts ) ) {
				$fonts = array();
			}

			return $fonts;

		}

		/*
		 * Get registered sidebars
		 */
		public function get_registered_sidebars() {

			global $wp_registered_sidebars;

			if ( ! is_array( $wp_registered_sidebars ) ) {
				$wp_registered_sidebars = array();
			}

			$sidebars = array();

			foreach ( $wp_registered_sidebars as $v ) {
				$sidebars[ $v['id'] ] = $v['name'];
			}

			$registered_custom_sidebars = array();
			foreach ( $this->to_settings['fields'] as $v ) {
				if ( $v['type'] == 'sidebar_create' ) {
					$value = array_key_exists( $v['id'], $this->to_saved ) ? $this->to_saved[ $v['id'] ] : ( array_key_exists( 'std', $v ) ? $v['std'] : array() );
					if ( is_array( $value ) ) {
						foreach ( $value as $i => $j ) {
							$k = self::get_custom_sidebar_prefix() . sanitize_key( $i );
							$registered_custom_sidebars[ $k ] = $j;
							if ( ! array_key_exists( $k, $sidebars ) ) {
								$sidebars[ $k ] = $j;
							}
						}
					}
				}
			}

			foreach ( $sidebars as $k => $v ) {
				if ( strpos( $k, self::get_custom_sidebar_prefix() ) === 0 && ! array_key_exists( $k, $registered_custom_sidebars ) ) {
					unset( $sidebars[ $k ] );
				}
			}

			return $sidebars;

		}

		/*
		 * Get custom sidebar id by given key
		 */
		public static function get_custom_sidebar_prefix() {

			return 'mco-custom-sidebar-';

		}

		/*
		 * Move all "advanced" metaboxes above the default editor
		 */
		public function move_advanced_meta_boxes() {

			global $post, $wp_meta_boxes;

			do_meta_boxes( get_current_screen(), 'advanced', $post );

			unset( $wp_meta_boxes[ get_post_type( $post ) ]['advanced'] );

		}

		/*
		 * Add classes to the metabox
		 */
		public function postbox_class( $args ) {

			$class = array();

			if ( ! is_array( $args ) ) {
				return $class;
			}

			$class = array( 'mco-postbox' );

			if ( array_key_exists( 'condition', $args ) && is_string( $args['condition'] ) ) {

				$current_opts = array(
					'post_format'   => 'standard',
					'page_template' => 'default'
				);

				preg_match( '#('. implode( '|', array_keys( $current_opts ) ) .')\:is\((.*)?\)#', $args['condition'], $condition );

				if ( ! empty( $condition ) ) {

					global $pagenow;

					if ( $pagenow == 'post.php' ) {

						$post_id = array_key_exists( 'post', $_GET ) ? intval( $_GET['post'] ) : ( array_key_exists( 'post', $_POST ) ? $_POST['post'] : '' );

						$current_opts['post_format'] = get_post_format( $post_id );
						if ( empty( $current_opts['post_format'] ) ) {
							$current_opts['post_format'] = 'standard';
						}

						$current_opts['page_template'] = get_page_template_slug( $post_id );
						if ( empty( $current_opts['page_template'] ) ) {
							$current_opts['page_template'] = 'default';
						}

					}

					if ( $current_opts[ $condition[1] ] != $condition[2] ) {
						$class[] = 'mco-hide';
					}

					$condition[1] = str_replace( '_', '-', $condition[1] );

					$class[] = 'cond-'. sanitize_html_class( $condition[1] );
					$class[] = 'visible-'. sanitize_html_class( $condition[1] ) .'-'. sanitize_title_with_dashes( $condition[2] );

				}

			}

			return $class;

		}

	}
}