<?php

if ( ! function_exists( 'sanfrancisco_comments_cb' ) ) {
	/**
	 * Comment list callback.
	 *
	 * @since   1.0.0
	 * @param   object $comment
	 * @param   array $args
	 * @param   int $depth
	 */
	function sanfrancisco_comments_cb( $comment, $args, $depth ) {

		$GLOBALS['comment'] = $comment;

		?>
		<li <?php comment_class( empty( $args['has_children'] ) ? '' : 'parent' ); ?>>
			<div id="comment-<?php comment_ID(); ?>" class="sf-comment-body">
				<footer class="sf-comment-meta">
					<div class="sf-comment-author vcard">
						<?php
						if ( 0 != $args['avatar_size'] ) {
							echo get_avatar( $comment, $args['avatar_size'] );
						}
						?>
						<b class="fn"><?php echo get_comment_author_link(); ?></b>
					</div>
					<div class="sf-comment-metadata">
						<a href="<?php echo esc_url( get_comment_link( $comment->comment_ID, $args ) ); ?>" itemprop="url">
							<time itemprop="commentTime" datetime="<?php comment_time( 'c' ); ?>" title="<?php comment_time( 'F j, Y g:i a' ); ?>"><?php echo esc_html( get_comment_date() ); ?></time>
						</a>
						<?php edit_comment_link( esc_html__( 'Edit', 'sanfrancisco' ), '  ', '' ); ?>
					</div>
				</footer>
				<div class="sf-comment-content">
					<?php if ( $comment->comment_approved == '0' ): ?>
					<div class="alert alert-warning">
						<?php esc_html_e( 'Your comment is awaiting moderation.', 'sanfrancisco' ); ?>
					</div>
					<?php endif; ?>
					<p><?php echo nl2br( get_comment_text() ); ?></p>
				</div>
				<div class="sf-comment-reply">
					<?php
					comment_reply_link( array_merge( $args, array(
						'reply_text' => esc_html__( 'Reply', 'sanfrancisco' ),
						'depth'     => $depth,
						'max_depth' => $args['max_depth']
					) ) );
					?>
				</div>
			</div>
	<?php

	}
}

if ( ! function_exists( 'sanfrancisco_comments_nav' ) ) {
	/**
	 * Comments navigation.
	 *
	 * @since   1.0.0
	 * @param   string $position
	 */
	function sanfrancisco_comments_nav( $position = 'top' ) {

		if ( get_comment_pages_count() > 1 && get_option( 'page_comments' ) ) : ?>
			<nav class="sf-comments-nav sf-comments-nav-<?php echo sanitize_html_class( $position ); ?>">

				<ul>
					<?php

					if ( $prev_link = get_previous_comments_link( esc_html__( '&laquo; Older Comments', 'sanfrancisco' ) ) ) {
						printf( '<li class="nav-previous">%s</li>', $prev_link );
					}

					if ( $next_link = get_next_comments_link( esc_html__( 'Newer Comments &raquo;', 'sanfrancisco' ) ) ) {
						printf( '<li class="nav-next">%s</li>', $next_link );
					}

					?>
				</ul>

			</nav><!-- .comments-nav -->
		<?php endif;

	}
}

if ( ! function_exists( 'sanfrancisco_logo' ) ) {
	/**
	 * Output the logo.
	 *
	 * @since   1.0.0
	 * @param   string $position
	 */
	function sanfrancisco_logo( $position ) {

		if ( ! in_array( $position, array( 'header', 'footer' ) ) ) {
			return;
		}

		$options = sanfrancisco_get_options( array(
			'display_'. $position .'_logo_as_text',
			$position .'_logo_text',
			$position .'_logo_url',
			$position .'_logo_retina_url',
			'display_'. $position .'_logo_mobile_as_text',
			$position .'_logo_mobile_text',
			$position .'_logo_mobile_url',
			$position .'_logo_mobile_retina_url'
		) );

		$class = ( $position == 'footer' ) ? 'sf-footer-brand' : 'navbar-brand';

		echo '<a href="'. esc_url( home_url( '/' ) ) .'" class="'. sanitize_html_class( $class ) .'">';

		foreach ( array( '', 'mobile_' ) as $k ) {

			$span_class = array(
				'sf-logo',
				sanitize_html_class( 'sf-'. $position .'-logo'. ( $k == 'mobile_' ? '-mobile' : '' ) ),
				( $k == 'mobile_' ) ? 'visible-xs' : 'hidden-xs'
			);

			echo '<span class="'. esc_attr( implode( ' ', $span_class ) ) .'">';

			if ( $options['display_'. $position .'_logo_'. $k .'as_text'] == 'true' ) {
				echo stripcslashes( esc_html( $options[ $position .'_logo_'. $k .'text' ] ) );
			} else {

				$img_attr = array();
				$logo_retina = sanfrancisco_get_image_url( $options[ $position .'_logo_'. $k .'retina_url' ], 'full', false );

				if ( ! empty( $logo_retina ) ) {
					$img_attr['data-rjs'] = $logo_retina;
				}

				sanfrancisco_image_tag( $options[ $position .'_logo_'. $k .'url' ], 'full', $img_attr, true );

			}

			echo '</span>';

		}

		echo '</a>';

	}
}

if ( ! function_exists( 'sanfrancisco_section_id' ) ) {
	/**
	 * Generate section id.
	 *
	 * @since   1.0.0
	 * @param   array $section
	 * @return  string
	 */
	function sanfrancisco_section_id( $section ) {

		return 'sf-section-'. sanitize_key( $section['id'] );

	}
}

if ( ! function_exists( 'sanfrancisco_module_id' ) ) {
	/**
	 * Generate module id.
	 *
	 * @since   1.0.0
	 * @param   array $module
	 * @return  string
	 */
	function sanfrancisco_module_id( $module ) {

		return 'sf-module-'. sanitize_key( $module['id'] );

	}
}

if ( ! function_exists( 'sanfrancisco_section_class' ) ) {
	/**
	 * Generate section class.
	 *
	 * @since   1.0.0
	 * @param   array $section
	 * @return  string
	 */
	function sanfrancisco_section_class( $section ) {

		$classes = array( 'sf-section', 'sf-section-'. sanitize_key( $section['settings']['layout'] ) );

		if ( $section['settings']['layout'] == 'ws' ) {
			$classes[] = 'sf-section-sidebar-'. sanitize_key( $section['settings']['sidebar_position'] );
			if ( $section['settings']['sidebar_position'] != 'none' ) {
				$classes[] = 'sf-section-has-sidebar';
			}
		}

		if ( array_key_exists( 'class_name', $section['settings'] ) && ! empty( $section['settings']['class_name'] ) ) {
			$classes = array_merge( $classes, explode( ' ', $section['settings']['class_name'] ) );
		}

		if ( array_key_exists( 'visibility', $section['settings'] ) && is_array( $section['settings']['visibility'] ) ) {
			foreach ( array( 'xs', 'sm', 'md', 'lg' ) as $k ) {
				$classes[] = ( in_array( 'visible-'. $k, $section['settings']['visibility'] ) ? 'visible-' : 'hidden-' ) . $k;
			}
		}

		return sanfrancisco_class( $classes );

	}
}

if ( ! function_exists( 'sanfrancisco_module_class' ) ) {
	/**
	 * Generate module class.
	 *
	 * @since   1.0.0
	 * @param   array $module
	 * @return  string
	 */
	function sanfrancisco_module_class( $module ) {

		$classes = array( 'sf-module', sanitize_html_class( 'sf-module-'. str_replace( '_', '-', $module['layout'] ) ) );

		if ( in_array( $module['layout'], array( 'grid_posts', 'featured_posts' ) ) ) {

			$classes[] = 'sf-entries';
			$classes[] = sanitize_html_class( 'sf-entries-'. $module['layout'] );

			if ( $module['layout'] == 'grid_posts' ) {

				$posts_layout = '';
				if ( array_key_exists( 'gp_posts_layout', $module ) ) {
					$posts_layout = $module['gp_posts_layout'];
				} else if ( array_key_exists( 'posts_layout', $module ) ) {
					$posts_layout = $module['posts_layout'];
				}

				preg_match( '#^(vertical|horizontal)_([0-9]{1})_col$#', $posts_layout, $match );

				if ( ! empty( $match ) ) {
					$classes[] = sanitize_html_class( 'sf-entries-col-'. ( empty( $match ) ? '1' : $match[2] ) );
					$classes[] = sanitize_html_class( 'sf-entries-'. $match[1] );
				}

			}

		} else if ( $module['layout'] == 'fancy_title' ) {

			$classes[] = 'sf-fancy-title';

		}

		if ( array_key_exists( 'class_name', $module ) && ! empty( $module['class_name'] ) ) {
			$classes = array_merge( $classes, explode( ' ', $module['class_name'] ) );
		}

		if ( array_key_exists( 'visibility', $module ) && ! empty( $module['visibility'] ) && is_array( $module['visibility'] ) ) {
			$classes = array_merge( $classes, $module['visibility'] );
		}

		return sanfrancisco_class( $classes );

	}
}

if ( ! function_exists( 'sanfrancisco_module_title_class' ) ) {
	/**
	 * Generate module title class.
	 *
	 * @since   1.0.0
	 * @param   $module
	 * @return  string
	 */
	function sanfrancisco_module_title_class( $module ) {

		$classes = array( 'sf-module-title' );

		if ( ! array_key_exists( 'title_position', $module ) ) {
			$module['title_position'] = 'center';
		}

		$classes[] = 'sf-'. sanitize_html_class( $module['title_position'] );

		if ( ! array_key_exists( 'title_has_bar', $module ) || $module['title_has_bar'] != 'false' ) {
			$classes[] = 'sf-has-line';
		}

		return sanfrancisco_class( $classes );

	}
}

if ( ! function_exists( 'sanfrancisco_nav_class' ) ) {
	/**
	 * Generate nav bar class.
	 *
	 * @since   1.0.0
	 * @param   array $classes
	 * @return  string
	 */
	function sanfrancisco_nav_class( $classes = array() ) {

		if ( ! is_array( $classes ) ) {
			$classes = array();
		}

		$options = sanfrancisco_get_options( array(
			'header_style',
			'sticky_header'
		) );
		
		$classes[] = 'navbar';
		$classes[] = 'navbar-style-'. str_replace( '_', '-', $options['header_style'] );

		if ( $options['sticky_header'] != 'false' ) {
			$classes[] = 'navbar-sticky';
		}

		return sanfrancisco_class( $classes );

	}
}

if ( ! function_exists( 'sanfrancisco_sidebar_class' ) ) {
	/**
	 * Generate sidebar class.
	 *
	 * @since   1.0.0
	 * @return  string
	 */
	function sanfrancisco_sidebar_class() {

		$classes = array( 'sf-sidebar' );

		if ( sanfrancisco_get_option( 'sticky_sidebar' ) != 'false' ) {
			$classes[] = 'sf-sidebar-sticky';
		}

		return sanfrancisco_class( $classes );

	}
}

if ( ! function_exists( 'sanfrancisco_singular_header' ) ) {
	/**
	 * Output singular post header. (Post title, excerpt and meta data)
	 *
	 * @since   1.0.1
	 * @param   array $args   module args
	 * @param   string $wrap_tag
	 */
	function sanfrancisco_singular_header( $args = array(), $wrap_tag = 'div' ) {

		global $post;

		if ( ! is_singular() ) {
			return;
		}

		if ( ! is_array( $args ) ) {
			$args = array();
		}

		$position = 'article';
		if ( array_key_exists( 'settings', $args ) ) {
			$position = 'before_section';
			if ( ! array_key_exists( 'modules', $args ) || ! is_array( $args['modules'] ) || ! array_key_exists( 'post', $args['modules'] ) ) {
				return;
			}
			$args = $args['modules']['post'];
		}

		$display_featured_image = ( ! array_key_exists( 'display_featured_image', $args ) || $args[ 'display_featured_image' ] != 'false' );

		$meta_data = array();

		if ( array_key_exists( 'meta_data', $args ) && is_array( $args['meta_data'] ) ) {
			$meta_data = $args['meta_data'];
		}

		if ( ! array_key_exists( 'display_title', $args ) || $args['display_title'] != 'false' ) {
			$meta_data[] = 'title';
		}

		if ( has_excerpt() && ( ! array_key_exists( 'display_title_subline', $args ) || $args['display_title_subline'] != 'false' ) ) {
			$meta_data[] = 'title_subline';
		}

		$featured_image_args = array(
			'flags'          => array(),
			'thumbnail_size' => 'sf-700-525',
			'style'          => ( array_key_exists( 'featured_image_style', $args ) && ! empty( $args[ 'featured_image_style' ] ) ) ? $args[ 'featured_image_style' ] : 'cl'
		);

		if ( ! in_array( 'display_credit_line', $args ) || $args['display_credit_line'] != 'false' ) {
			$featured_image_args['flags'][] = 'credit_line';
		}

		if ( ( $featured_image_args['style'] == 'cl' && $position == 'before_section' )
			|| ( $featured_image_args['style'] != 'cl' && $position != 'before_section' ) ) {
			return;
		}

		if ( $position == 'before_section' ) {
			$featured_image_args['thumbnail_size'] = 'sf-1140-855';
		}

		$meta_data_int = array_intersect( array(
			'category',
			'views',
			'likes',
			'comments',
			'byline',
			'title',
			'title_subline'
		), $meta_data );

		// Return empty if all required singular header items are disabled.
		if ( empty( $meta_data_int ) ) {
			return;
		}

		echo '<' . $wrap_tag . ' class="sf-entry-header">';

		// Output trending icon if enabled.
		if ( in_array( 'trending_icon', $meta_data_int ) && sanfrancisco_post_is_trending( $post ) ) {
			echo '<div class="sf-entry-flags"><span class="sf-entry-flag sf-entry-flag-trending"><i class="fa fa-bolt"></i></span></div>';
		}

		// Output breadcrumb if enabled.
		if ( sanfrancisco_get_option( 'display_breadcrumb' ) == 'true' ) {
			sanfrancisco_breadcrumb();
		}

		$meta_data_int = array_intersect( array(
			'category',
			'views',
			'likes',
			'comments',
			'byline'
		), $meta_data );

		// Output meta data if enabled.
		if ( ! empty( $meta_data_int ) ) {

			echo '<ul class="sf-entry-meta clearfix">';

			if ( in_array( 'category', $meta_data ) ) {
				echo '<li class="sf-entry-meta-category">' . sanfrancisco_post_categories( $post, array( 'class' => 'sf-btn sf-btn-gray active' ), false ) . '</li>';
			}

			if ( in_array( 'byline', $meta_data ) ) {
				echo '<li class="sf-entry-meta-byline">' . sanfrancisco_byline( $post, false ) . '</li>';
			}

			if ( in_array( 'views', $meta_data ) ) {
				echo '<li class="sf-entry-meta-views"><span class="fa fa-eye"></span> <span class="sf-count">' . sanfrancisco_post_views( $post, false ) . '</span></li>';
			}

			if ( in_array( 'likes', $meta_data ) ) {
				echo '<li class="sf-entry-meta-likes"><span class="fa fa-heart-o"></span> <span class="sf-count">' . sanfrancisco_post_likes( 'like', $post, false ) . '</span></li>';
			}

			if ( in_array( 'comments', $meta_data ) ) {
				echo '<li class="sf-entry-meta-comments"><span class="fa fa-comment-o"></span> <span class="sf-count">' . sanfrancisco_post_comment_count( $post, false ) . '</span></li>';
			}

			echo '</ul>';

		}

		// Output post title if enabled.
		if ( in_array( 'title', $meta_data ) ) {

			$post_title = $post->post_title;
			if ( empty( $post_title ) ) {
				$post_title = esc_html__( '(no title)', 'sanfrancisco' );
			}

			if ( ! empty( $post_title ) ) {
				echo '<h1 class="sf-entry-title" itemprop="headline">' . $post_title . '</h1>';
			}

		}

		if ( in_array( 'title_subline', $meta_data ) ) {
			echo '<h2 class="sf-entry-subtitle">' . sanfrancisco_get_excerpt( $post ) . '</h2>';
		}

		echo '</' . $wrap_tag . '>';

		if ( $display_featured_image && $position != 'before_section' ) {
			sanfrancisco_featured_media( $post, $featured_image_args );
		}

	}
}
add_action( 'sanfrancisco_before_section', 'sanfrancisco_singular_header' );

if ( ! function_exists( 'sanfrancisco_featured_media' ) ) {
	/**
	 * Output post featured media
	 *
	 * @since   1.0.0
	 * @param   object  $post
	 * @param   array   $args
	 * @param   bool    $echo
	 * @return  string
	 */
	function sanfrancisco_featured_media( $post = null, $args = array(), $echo = true ) {

		$post = get_post( $post );
		if ( ! $post ) {
			return '';
		}

		if ( ! is_array( $args ) ) {
			$args = array();
		}

		$args = wp_parse_args( $args, array(
			'zoom_on_hover' => false,
			'is_teaser'     => false,
			'flags'         => array(),
			'style'         => 'cl'
		) );

		$post_format            = get_post_format( $post );
		$gif_image_id           = get_post_meta( $post->ID, 'sanfrancisco_gif_featured_image', true );
		$thumbnail_id           = empty( $gif_image_id ) ? get_post_thumbnail_id( $post ) : $gif_image_id;
		$credit_line            = get_post_meta( $thumbnail_id, 'sanfrancisco_credit_line', true );
		$featured_image_size    = array_key_exists( 'thumbnail_size', $args ) ? $args['thumbnail_size'] : 'sf-700-525';
		$featured_media_html    = $featured_image_html = $media_icon = '';

		if ( $args['style'] != 'cl' ) {
			$featured_image_size = 'sf-1140-570';
		}

		if ( ! empty( $gif_image_id ) && $post_format != 'gallery' ) {
			$featured_image_size = 'full';
		}

		if ( $post_format == 'gallery' ) {

			$gallery_images = get_post_meta( $post->ID, 'sanfrancisco_gallery_images', true );

			if ( is_array( $gallery_images ) ) {

				$featured_media_html .= '<figure class="sf-entry-featured-media"><div class="owl-carousel" data-items="1" data-loop="true" data-dots="'. ( $args['is_teaser'] ? 'false' : 'true' ) .'" data-controls=".sf-carousel-controls">';
				foreach ( $gallery_images as $k => $v ) {
					$featured_media_html .= '<div class="item">';

					$featured_media_html .= sanfrancisco_image_tag( $v, $featured_image_size, array(), false );

					if ( ! $args['is_teaser'] && get_post_meta( $post->ID, 'sanfrancisco_gallery_display_caption', true ) == 'true' ) {
						$attachment = get_post( $v );
						if ( $attachment->post_excerpt != null ) {
							$featured_media_html .= '<div class="sf-caption">'. esc_html( $attachment->post_excerpt ) .'</div>';
						}
					}

					$featured_media_html .= '</div>';
				}
				$featured_media_html .= '</div>
					<div class="sf-carousel-controls">
						<span class="sf-control"><span class="fa fa-angle-left"></span></span>
						<span class="sf-control"><span class="fa fa-angle-right"></span></span>
					</div></figure>';

			}

			if ( ! $args['is_teaser'] && get_post_meta( $post->ID, 'sanfrancisco_gallery_override_featured_image', true ) != 'false' ) {
				$thumbnail_id = '';
			}

			if ( $args['is_teaser'] ) {

				if ( get_post_meta( $post->ID, 'sanfrancisco_gallery_override_teaser_image', true ) != 'false' || $args['style'] != 'cl' ) {
					$thumbnail_id = '';
				} else {
					$featured_media_html = '';
				}

			}

		} else if ( in_array( $post_format, array( 'video', 'audio' ) ) ) {

			if ( get_post_meta( $post->ID, 'sanfrancisco_'. $post_format .'_display_media_icon', true ) != 'false' ) {
				$media_icon = '<span class="sf-entry-media-icon"><i class="fa fa-'. ( $post_format == 'video' ? 'play' : 'music' ) .'"></i></span>';
			}

			if ( ! $args['is_teaser'] ) {

				$media_type = get_post_meta( $post->ID, 'sanfrancisco_'. $post_format .'_type', true );
				if ( ! in_array( $media_type, array( 'external', 'internal' ) ) ) {
					$media_type = 'external';
				}

				$media_content = get_post_meta( $post->ID, 'sanfrancisco_'. $post_format .'_'. ( $media_type == 'external' ? 'embed_code' : 'url' ), true );
				if ( ! empty( $media_content ) && function_exists( 'hybrid_media_grabber' ) ) {
					$featured_media_html .= '<figure class="sf-entry-featured-media">';
					$featured_media_html .= hybrid_media_grabber( array(
						'type'          => $post_format,
						'split_media'   => true,
						'content'       => $media_content,
						'width'         => 1140
					) );
					$featured_media_html .= '</figure>';
				}

				if ( get_post_meta( $post->ID, 'sanfrancisco_'. $post_format .'_override_featured_image', true ) != 'false' || $args['style'] != 'cl' ) {
					$thumbnail_id = '';
				}

			}

		}

		if ( ! empty( $thumbnail_id ) ) {

			if ( $args['is_teaser'] ) {
				$featured_image_html .= '<a href="'. esc_url( get_permalink( $post ) ) .'">' . $media_icon;
			}

			$featured_image_html .= sanfrancisco_image_tag( $thumbnail_id, $featured_image_size, array(
				'class' => 'sf-featured-image'
			), false );

			if ( $args['is_teaser'] ) {
				$featured_image_html .= '</a>';
			}

		}

		if ( ! empty( $featured_image_html ) ) {

			$featured_media_html .= '<figure class="sf-entry-featured-media ' . ( $args['zoom_on_hover'] ? ' sf-zoom-hover' : '' ) . '">' . $featured_image_html;

			if ( ! empty( $args['flags'] ) ) {

				$featured_media_html .= '<div class="sf-entry-flags">';

				if ( ! empty( $credit_line ) && in_array( 'credit_line', $args['flags'] ) ) {
					$featured_media_html .= '<span class="sf-entry-flag sf-entry-flag-creditline">' . esc_html( $credit_line ) . '</span>';
				}

				if ( in_array( 'trending_icon', $args['flags'] ) ) {
					$featured_media_html .= '<span class="sf-entry-flag sf-entry-flag-trending"><i class="fa fa-bolt"></i></span>';
				}

				if ( in_array( 'category', $args['flags'] ) ) {
					$featured_media_html .= '<span class="sf-entry-flag sf-entry-flag-category">' . sanfrancisco_post_categories( $post, array(), false ) . '</span>';
				}

				$featured_media_html .= '</div>';

			}

			$featured_media_html .= '</figure>';

		}

		if ( $args['style'] != 'cl' ) {
			$featured_media_html = '<div class="sf-entry-fm-wide sf-entry-fm-' . esc_attr( $args['style'] ) . '"><div class="container">' . $featured_media_html . '</div></div>';
		}

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_featured_media_html', $featured_media_html );
		}

		return $featured_media_html;

	}
}

if ( ! function_exists( 'sanfrancisco_image_tag' ) ) {
	/**
	 * Get attachment image tag.
	 *
	 * @since   1.0.0
	 * @param   int $attachment_id
	 * @param   string $size
	 * @param   array $attr
	 * @param   bool $echo
	 * @return  string
	 */
	function sanfrancisco_image_tag( $attachment_id, $size = 'full', $attr = array(), $echo = true ) {

		$image_tag = '';
		$image = wp_get_attachment_image_src( $attachment_id, $size, false );
		$image_size = sanfrancisco_get_image_sizes( $size );

		if ( $image ) {

			list( $src, $width, $height ) = $image;

			$hwstring = image_hwstring( $width, $height );

			$size_class = $size;
			if ( is_array( $size_class ) ) {
				$size_class = join( 'x', $size_class );
			}

			$img_datas = array(
				'50'  => 'R0lGODlhAgABAIAAAP///wAAACH5BAEAAAAALAAAAAACAAEAAAICBAoAOw==',
				'75'  => 'R0lGODlhBAADAIAAAAAAAAAAACH5BAEAAAAALAAAAAAEAAMAAAIDhI9WADs=',
				'88'  => 'R0lGODlhFgAZAIAAAAAAAAAAACH5BAEAAAAALAAAAAAWABkAAAIVhI+py+0Po5y02ouz3rz7D4biSGoFADs=',
				'100' => 'R0lGODlhAQABAIAAAAAAAAAAACH5BAEAAAAALAAAAAABAAEAAAICRAEAOw=='
			);

			$src_placeholder = '';
			if ( array_key_exists( 'ratio', $image_size ) && array_key_exists( $image_size['ratio'], $img_datas ) ) {
				$src_placeholder = 'data:image/gif;base64,' . $img_datas[ $image_size['ratio'] ];
			}

			$attachment = get_post( $attachment_id );
			$default_attr = array(
				'data-src' => esc_url( $src ),
				'src'      => $src_placeholder,
				'class'    => "attachment-$size_class size-$size_class lazyload agg-fade-in",
				'alt'      => trim( strip_tags( get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ) ) ) // Use Alt field first
			);

			if ( empty( $default_attr['src'] ) ) {
				$default_attr['src'] = $default_attr['data-src'];
				unset( $default_attr['data-src'] );
			}

			if ( empty( $default_attr['alt'] ) ) {
				$default_attr['alt'] = trim( strip_tags( $attachment->post_excerpt ) ); // If not, Use the Caption
			}

			if ( empty( $default_attr['alt'] ) ) {
				$default_attr['alt'] = trim( strip_tags( $attachment->post_title ) ); // Finally, use the title
			}

			if ( array_key_exists( 'class', $attr ) ) {
				if ( ! is_string( $attr['class'] ) ) {
					$attr['class'] = '';
				} else {
					$attr['class'] .= ' '. $default_attr['class'];
				}
			}

			$attr = wp_parse_args( $attr, $default_attr );

			// Generate 'srcset' and 'sizes' if not already present.
			if ( empty( $attr['srcset'] ) ) {

				$image_meta = get_post_meta( $attachment_id, '_wp_attachment_metadata', true );

				if ( is_array( $image_meta ) ) {
					
					$size_array = array( absint( $width ), absint( $height ) );
					$srcset = sanfrancisco_calculate_image_srcset( $size_array, $src, $image_meta, $attachment_id );
					$sizes = sanfrancisco_calculate_image_sizes( $size_array, $src, $image_meta, $attachment_id );

					if ( $srcset && ( $sizes || ! empty( $attr['sizes'] ) ) ) {
						$attr['data-srcset'] = $srcset;

						if ( empty( $attr['sizes'] ) ) {
							$attr['data-sizes'] = $sizes;
						}
					}
				}

			}

			$attr = array_map( 'esc_attr', $attr );
			$image_tag = rtrim("<img $hwstring");
			foreach ( $attr as $name => $value ) {
				$image_tag .= " $name=" . '"' . $value . '"';
			}
			$image_tag .= ' />';

		}

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_image_tag', $image_tag );
		}

		return $image_tag;

	}
}

if ( ! function_exists( 'sanfrancisco_bg_srcset_attr' ) ) {
	/**
	 * Responsive background image srcset attributes.
	 *
	 * @since   1.0.0
	 * @param   int $attachment_id
	 * @param   string $size
	 */
	function sanfrancisco_bg_srcset_attr( $attachment_id, $size = 'full' ) {

		$attr_output = '';
		$image = wp_get_attachment_image_src( $attachment_id, $size, false );

		if ( $image ) {

			list( $src, $width, $height ) = $image;

			$srcset = $sizes = false;

			if ( $size != 'full' ) {

				$image_meta = get_post_meta( $attachment_id, '_wp_attachment_metadata', true );

				if ( is_array( $image_meta ) ) {
					$size_array = array( absint( $width ), absint( $height ) );
					$srcset = sanfrancisco_calculate_image_srcset( $size_array, $src, $image_meta, $attachment_id );
					$sizes = sanfrancisco_calculate_image_sizes( $size_array, $src, $image_meta, $attachment_id );
				}

			}

			if ( ! $srcset || ! $sizes ) {
				$srcset = $src;
				$sizes = 'auto';
			}

			if ( $srcset && $sizes ) {
				$attr_output .= ' data-bgset="'. $srcset .'" data-sizes="'. $sizes .'"';
			}

		}

		echo apply_filters( 'sanfrancisco_bg_srcset_attr', $attr_output );

	}
}

if ( ! function_exists( 'sanfrancisco_get_excerpt' ) ) {
	/**
	 * Get post excerpt.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @param   mixed $length
	 * @return  string
	 */
	function sanfrancisco_get_excerpt( $post = null, $length = false ) {

		$post = get_post( $post );
		if ( ! $post || post_password_required() ) {
			return '';
		}

		$post_excerpt = $post->post_excerpt;
		if ( $post_excerpt == null ) {
			$post_excerpt = $post->post_content;
		}

		$post_excerpt = strip_shortcodes( $post_excerpt );
		$post_excerpt = str_replace( ']]>', ']]&gt;', $post_excerpt );
		$post_excerpt = strip_tags( $post_excerpt );
		$post_excerpt = preg_replace( '~http(s)?://[^\s]*~i', '', $post_excerpt );

		if ( $length ) {

			$length = intval( $length ) * 4;

			if ( $length < strlen( $post_excerpt ) ) {

				$post_excerpt = mb_substr( $post_excerpt, 0, $length, 'UTF-8' ) . '&hellip;';

				//if ( ! is_singular() || is_page_template( 'page-sections.php' ) ) {
				//	$post_excerpt .= '<span class="sf-more-text"><a href="'. esc_url( get_permalink( $post ) ) .'">' . esc_html__( 'More &raquo;', 'sanfrancisco' ) . '</a></span>';
				//}

			}

		}

		// Allow post excerpt to be filtered before output
		$post_excerpt = apply_filters( 'sanfrancisco_excerpt', $post_excerpt );

		return $post_excerpt;

	}
}

if ( ! function_exists( 'sanfrancisco_excerpt' ) ) {
	/**
	 * Output post excerpt.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @param   mixed $length
	 */
	function sanfrancisco_excerpt( $post = null, $length = false ) {

		echo sanfrancisco_get_excerpt( $post, $length );

	}
}

if ( ! function_exists( 'sanfrancisco_post_categories' ) ) {
	/**
	 * Get post categories.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @param   int $limit number of categories to retrieve
	 * @param   array $attr
	 * @param   bool $echo
	 * @param   bool $link  true if you want to add link to category names
	 * @param   bool $icon  true if you want to add icon at the beginning of category names
	 * @return  string
	 */
	function sanfrancisco_post_categories( $post = null, $attr = array(), $echo = true, $limit = 1, $link = true, $icon = false ) {

		$post = get_post( $post );
		if ( ! $post ) {
			return '';
		}

		if ( ! is_array( $attr ) ) {
			$attr = array();
		}

		$html = $href = $name = '';
		$tag = $link ? 'a' : 'span';
		$post_type = get_post_type();
		$post_taxonomies = apply_filters( 'sanfrancisco_category_tax_names', array() );

		if ( array_key_exists( $post_type, $post_taxonomies ) ) {

			$post_terms = wp_get_post_terms( $post->ID, $post_taxonomies[ $post_type ] );

			$i = 1;
			foreach ( $post_terms as $k => $v ) {

				$name = $v->name;
				$href = get_category_link( $v->term_id );

				if ( $i >= $limit ) {
					break;
				}

				$i++;

			}

			$html = '<'. $tag .' '. ( $tag == 'a' ? 'href="'. esc_url( $href ) .'"' : '' );

			if ( ! empty( $attr ) ) {
				foreach ( $attr as $k => $v ) {
					$html .= " $k=" . '"' . $v . '"';
				}
			} else {
				$html = rtrim( $html );
			}

			$html .= '>'. ( $icon ? '<span class="fa fa-'. sanitize_key( $icon ) .'"></span> ' : '' ) . esc_html( $name ) .'</'. $tag .'>';

		}

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_post_categories_output', $html );
		}

		return apply_filters( 'sanfrancisco_post_categories_output', $html );

	}
}

if ( ! function_exists( 'sanfrancisco_post_tags' ) ) {
	/**
	 *  Get post tags.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @param   array $attr
	 * @param   bool $echo
	 * @param   int $limit
	 * @return  string
	 */
	function sanfrancisco_post_tags( $post = null, $limit = 3, $attr = array(), $echo = true ) {

		$post = get_post( $post );
		if ( ! $post ) {
			return '';
		}

		if ( ! is_array( $attr ) ) {
			$attr = array();
		}

		// theme check pass
		get_the_tag_list();
		$html = '';
		$post_type = get_post_type();
		$post_taxonomies = apply_filters( 'sanfrancisco_tag_tax_names', array() );

		if ( array_key_exists( $post_type, $post_taxonomies ) ) {

			$post_terms = wp_get_post_terms( $post->ID, $post_taxonomies[ $post_type ] );

			if ( ! empty( $post_terms ) ) {

				$html = '<p';

				foreach ( $attr as $k => $v ) {
					$html .= " $k=" . '"' . $v . '"';
				}

				$html .= '><span class="sf-label">'. esc_html__( 'Tags:', 'sanfrancisco' ) .'</span>';

				$i = 0;
				foreach ( $post_terms as $k => $v ) {
					$html .= '<span class="sf-tag"><a href="'. esc_url( get_category_link( $v->term_id ) ) .'">'. esc_html( $v->name ) .'</a></span>';
					$i++;
					if ( $i >= $limit ) {
						break;
					}
				}

				$html .= '</p>';

			}

		}

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_post_tags_output', $html );
		}

		return apply_filters( 'sanfrancisco_post_tags_output', $html );

	}
}

if ( ! function_exists( 'sanfrancisco_byline' ) ) {
	/**
	 * Output post byline meta data.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @param   bool $echo
	 * @param   bool $link  true if you want to add link to the author name
	 * @return  string
	 */
	function sanfrancisco_byline( $post = null, $echo = true, $link = true ) {

		$post = get_post( $post );
		if ( ! $post ) {
			return '';
		}

		$tag = $link ? 'a' : 'span';
		$author_id = $post->post_author;
		$avatar_size = is_singular() ? 32 : 26;

		$author_avatar = apply_filters( 'sanfrancisco_byline_avatar', get_avatar( $author_id, $avatar_size ) );

		$html = '<span class="sf-entry-author" itemscope itemtype="https://schema.org/Person">
			<'. $tag . ( $tag == 'a' ? ' href="'. esc_url( get_author_posts_url( $author_id ) ) .'"' : '' ) .' rel="author">'.
				$author_avatar .'<span itemprop="author">'. get_the_author_meta( 'display_name', $author_id ) .'</span>
			</'. $tag .'>
		</span>';

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_byline_output', $html );
		}

		return apply_filters( 'sanfrancisco_byline_output', $html );

	}
}

if ( ! function_exists( 'sanfrancisco_post_views' ) ) {
	/**
	 * Get number of post views.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @param   bool $echo
	 * @return  int
	 */
	function sanfrancisco_post_views( $post = null, $echo = true ) {

		$post_views = 0;

		$post = get_post( $post );
		if ( ! $post ) {
			return $post_views;
		}

		$post_views = get_post_meta( $post->ID, sanfrancisco_views_meta_key(), true );
		if ( empty( $post_views ) ) {
			$post_views = 0;
		}

		// Allow post views to be filtered before output
		$post_views = apply_filters( 'sanfrancisco_post_views', $post_views );

		$post_views = intval( $post_views );

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_post_views_output', $post_views );
		}

		return apply_filters( 'sanfrancisco_post_views_output', $post_views );

	}
}

if ( ! function_exists( 'sanfrancisco_post_likes' ) ) {
	/**
	 * Get number of post likes/unlikes.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @param   bool $echo
	 * @return  int
	 */
	function sanfrancisco_post_likes( $type = 'like', $post = null, $echo = true ) {

		$post_likes = 0;

		$post = get_post( $post );
		if ( ! $post ) {
			return $post_likes;
		}

		$meta_key_func = 'sanfrancisco_' . $type . 's_meta_key';
		$post_likes = get_post_meta( $post->ID, $meta_key_func(), true );
		if ( empty( $post_likes ) ) {
			$post_likes = 0;
		}

		// Allow post likes to be filtered before output
		$post_likes = apply_filters( 'sanfrancisco_post_' . $type . 's', $post_likes );
		$post_likes = intval( $post_likes );

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_post_likes_output', $post_likes );
		}

		return apply_filters( 'sanfrancisco_post_likes_output', $post_likes );

	}
}

if ( ! function_exists( 'sanfrancisco_post_comment_count' ) ) {
	/**
	 * Get number of post comments.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @param   bool $echo
	 * @return  int
	 */
	function sanfrancisco_post_comment_count( $post = null, $echo = true ) {

		$comment_count = 0;

		$post = get_post( $post );
		if ( ! $post ) {
			return $comment_count;
		}

		$comment_count = get_comment_count( $post->ID );
		$comment_count = ( is_array( $comment_count ) && array_key_exists( 'approved', $comment_count ) ) ? $comment_count['approved'] : 0;

		// Allow comment count to be filtered before output
		$comment_count = apply_filters( 'sanfrancisco_post_comment_count', $comment_count );

		$comment_count = intval( $comment_count );

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_post_comment_count_output', $comment_count );
		}

		return apply_filters( 'sanfrancisco_post_comment_count_output', $comment_count );

	}
}

if ( ! function_exists( 'sanfrancisco_like_button' ) ) {
	/**
	 * Output like button.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @param   bool $echo
	 * @return  string $html
	 */
	function sanfrancisco_like_button( $post = null, $echo = true ) {

		$html = '';

		$post = get_post( $post );
		if ( $post ) {

			$is_liked = sanfrancisco_post_is( 'liked', $post );
			$is_unliked = sanfrancisco_post_is( 'unliked', $post );

			?>
			<div class="sf-like-box">
				<div class="sf-col">
					<div class="sf-like-btn sf-like-btn-like<?php echo ( true == $is_liked ? ' active' : ( $is_unliked ? ' disabled' : '' ) ); ?>" data-type="like" data-disabled="<?php echo ( true == $is_unliked ) ? 'true' : 'false'; ?>" data-active="<?php echo ( true == $is_liked ) ? 'true' : 'false'; ?>" data-id="<?php echo intval( $post->ID ); ?>">
						<span class="sf-count"><?php sanfrancisco_post_likes( 'like', $post, true ); ?></span>
						<span class="sf-icon"><i class="fa fa-heart"></i></span>
						<span class="sf-text"><?php esc_html_e( 'I like it', 'sanfrancisco' ); ?></span>
					</div>
				</div>
				<div class="sf-col">
					<div class="sf-like-btn sf-like-btn-unlike<?php echo ( true == $is_unliked ? ' active' : ( $is_liked ? ' disabled' : '' ) ); ?>" data-type="unlike" data-disabled="<?php echo ( true == $is_liked ) ? 'true' : 'false'; ?>" data-active="<?php echo ( true == $is_unliked ) ? 'true' : 'false'; ?>" data-id="<?php echo intval( $post->ID ); ?>">
						<span class="sf-count"><?php sanfrancisco_post_likes( 'unlike', $post, true ); ?></span>
						<span class="sf-text"><?php esc_html_e( 'I don\'t like it', 'sanfrancisco' ); ?></span>
					</div>
				</div>
			</div>
			<?php
		}

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_like_button_output', $html );
		}

		return apply_filters( 'sanfrancisco_like_button_output', $html );

	}
}

if ( ! function_exists( 'sanfrancisco_related_posts' ) ) {
	/**
	 * Output related posts.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @param   array $module
	 * @param   bool $echo
	 * @return  string
	 */
	function sanfrancisco_related_posts( $post = null, $module, $echo = true ) {

		$post = get_post( $post );
		if ( ! $post ) {
			return '';
		}

		$html = '';
		$module_id = 'related-entries';

		$opts = sanfrancisco_get_options( array(
			'related_posts_layout',
			'related_posts_per_page',
			'related_posts_excluded_post_ids',
			'related_posts_orderby',
			'related_posts_order',
			'related_posts_meta_data',
			'related_posts_excerpt_length'
		) );

		$related_posts_ids = sanfrancisco_get_related_post_ids( $post, $opts['related_posts_per_page'] );

		if ( empty( $related_posts_ids ) ) {
			return '';
		}

		$args['section_layout'] = 'ws';
		if ( array_key_exists( 'section_layout', $module ) ) {
			$args['section_layout'] = $module['section_layout'];
		}

		$args = array(
			'settings' => array(
				'id' => $module_id,
				'layout' => $args['section_layout']
			),
			'modules' => array()
		);

		$args['modules'][ $module_id ] = array(
			'id'                => $module_id,
			'title'             => esc_html__( 'You May Also Like', 'sanfrancisco' ),
			'layout'            => 'grid_posts',
			'gp_posts_layout'   => $opts['related_posts_layout'],
			'ignore_sticky_posts' => true,
			'posts_per_page'    => $opts['related_posts_per_page'],
			'post_ids'          => $related_posts_ids,
			'excluded_post_ids' => ltrim( $opts['related_posts_excluded_post_ids'] .','. $post->ID, ',' ),
			'orderby'           => $opts['related_posts_orderby'],
			'order'             => $opts['related_posts_order'],
			'meta_data'         => $opts['related_posts_meta_data'],
			'excerpt_length'    => $opts['related_posts_excerpt_length'],
			'display_tags'      => 'false',
			'class_name'        => 'sf-related-entries',
			'meta_data_pos'     => 'bottom'
		);

		sanfrancisco_load_template( array(
			'template' => 'module',
			'prefix'   => 'loop',
			'args'     => $args
		) );

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_related_posts_output', $html );
		}

		return apply_filters( 'sanfrancisco_related_posts_output', $html );

	}
}

if ( ! function_exists( 'sanfrancisco_pagination' ) ) {
	/**
	 * Output grid posts pagination.
	 *
	 * @since   1.3.1
	 * @param   string $type
	 * @param   mixed $query
	 * @param   bool $echo
	 * @return  string $output
	 */
	function sanfrancisco_pagination( $type, $query = false, $echo = true ) {

		$html = '';

		if ( ! $query ) {
			global $wp_query;
			$query = $wp_query;
		}

		$is_ajax = ( ( defined( 'DOING_AJAX' ) && DOING_AJAX ) || is_page_template( 'page-sections.php' ) );
		$max_num_pages = is_single() ? 99999 : $query->max_num_pages;
		$current_page = is_single() ? 1 : max( 1, $query->query_vars['paged'] );

		if ( $type == 'numeric' ) {

			if ( is_singular() && ! is_page_template( 'page-sections.php' ) ) {

				$html .= wp_link_pages( array(
					'before'           => '',
					'after'            => '',
					'link_before'      => '<span class="page-numbers">',
					'link_after'       => '</span>',
					'next_or_number'   => 'number',
					'separator'        => '',
					'nextpagelink'     => esc_html__( 'Next &raquo;', 'sanfrancisco' ),
					'previouspagelink' => esc_html__( '&laquo; Prev', 'sanfrancisco' ),
					'pagelink'         => '%',
					'echo'             => false
				) );

			} else {

				$html_arr = paginate_links( array(
					'base'      => $is_ajax ? '#%#%' : str_replace( 99999, '%#%', esc_url( get_pagenum_link( 99999 ) ) ),
					'type'      => 'array',
					'format'    => '',
					'current'   => $current_page,
					'total'     => $max_num_pages,
					'prev_text' => esc_html__( '&laquo; Prev', 'sanfrancisco' ),
					'next_text' => esc_html__( 'Next &raquo;', 'sanfrancisco' )
				) );

				if ( is_array( $html_arr ) ) {

					foreach ( $html_arr as $k => $v ) {

						if ( $is_ajax ) {
							$v = str_replace( array( '<a ', '</a>' ), array( '<span ', '</span>' ), $v );
							$v = preg_replace( '#href\=(\'|\")\#([0-9]+)(\'|\")#', 'data-paged="$2"', $v );
						}

						$html .= $v;

					}

				}

			}



		} else if ( $type == 'prev_next' ) {

			if ( is_single() ) {

				$html .= get_previous_post_link( '%link',  sprintf( esc_html__( '%s Previous Post', 'sanfrancisco' ), '<span class="fa fa-chevron-left"></span>' ) );
				$html .= get_next_post_link( '%link', sprintf( esc_html__( 'Next Post %s', 'sanfrancisco' ), '<span class="fa fa-chevron-right"></span>' ) );

			} else {

				if ( $is_ajax ) {

					$prev_page = ( $current_page == 1 ) ? false : $current_page - 1;
					$next_page = ( $current_page == $max_num_pages ) ? false : $current_page + 1;

					if ( $next_page ) {
						$html .= '<span class="pg-btn" data-paged="' . intval( $next_page ) . '" rel="next">' . sprintf( esc_html__( 'Older Entries %s', 'sanfrancisco' ), '<span class="fa fa-chevron-right"></span>' ) . '</span>';
					}

					if ( $prev_page ) {
						$html .= '<span class="pg-btn" data-paged="' . intval( $prev_page ) . '" rel="prev">' . sprintf( esc_html__( '%s Newer Entries', 'sanfrancisco' ), '<span class="fa fa-chevron-left"></span>' ) . '</span>';
					}

				} else {

					$html .= get_next_posts_link( sprintf( esc_html__( 'Older Entries %s', 'sanfrancisco' ), '<span class="fa fa-chevron-right"></span>' ), $max_num_pages );
					$html .= get_previous_posts_link( sprintf( esc_html__( '%s Newer Entries', 'sanfrancisco' ), '<span class="fa fa-chevron-left"></span>' ) );

				}

			}

		} else if ( in_array( $type, array( 'load_more', 'infinite_scroll' ) ) ) {

			$next_page = ( $current_page == $max_num_pages ) ? false : $current_page + 1;

			if ( $next_page ) {

				$btn_text = ( $type == 'infinite_scroll' ) ? esc_html__( 'Loading...', 'sanfrancisco' ) : esc_html__( 'Load More', 'sanfrancisco' );

				$html .= '<span data-href="' . esc_url( get_pagenum_link( $next_page ) ) . '" class="pg-btn" data-load-page="' . intval( $next_page ) . '"><i class="fa fa-circle-o-notch"></i>' . $btn_text . '</span>';

			}

		} else {

			return $html;

		}

		if ( ! empty( $html ) ) {
			$html = '<div class="sf-pagination sf-pagination-' . esc_attr( str_replace( '_', '-', $type ) ) .'">' . $html . '</div>';
		}

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_breadcrumb_output', $html );
		}

		return apply_filters( 'sanfrancisco_breadcrumb_output', $html );

	}
}

if ( ! function_exists( 'sanfrancisco_breadcrumb' ) ) {
	/**
	 * Output breadcrumb.
	 *
	 * @since   1.0.0
	 * @param   bool $echo
	 * @return  string
	 */
	function sanfrancisco_breadcrumb( $echo = true ) {

		$html = '';
		$post_type = is_single() ? get_post_type() : null;

		if ( $post_type == 'post' || is_category() ) {

			if ( is_category() ) {

				$categories = array( get_queried_object() );

				$cat = $categories[0];
				while( $cat->category_parent != '0' ) {
					$cat = get_category( $cat->category_parent );
					array_unshift( $categories, $cat );
				}

			} else {

				$categories = get_the_category();

			}

			foreach ( $categories as $k => $v ) {

				$html .= '<li itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">
							<a href="'. esc_url( get_category_link( $v ) ) .'" itemscope itemtype="https://schema.org/Thing" itemprop="item">
								<span itemprop="name">'. esc_html( $v->name ) .'</span>
							</a>
							<meta itemprop="position" content="'. ( $k + 2 ) .'" />
						</li>';

			}

		} else if ( is_archive() || is_search() || is_page() ) {

			$title = is_search() ? esc_html__( 'Search Results', 'sanfrancisco' ) : esc_html( get_the_archive_title() );

			if ( is_author() ) {

				$title = esc_html__( 'Author Archive', 'sanfrancisco' );

			} else if ( is_year() ) {

				$title = esc_html__( 'Yearly Archive', 'sanfrancisco' );

			} else if ( is_month() ) {

				$title = esc_html__( 'Monthly Archive', 'sanfrancisco' );

			} else if ( is_day() ) {

				$title = esc_html__( 'Daily Archive', 'sanfrancisco' );

			} else if ( is_tag() ) {

				$title = esc_html__( 'Tag Archive', 'sanfrancisco' );

			} else if ( is_page() ) {

				global $post;

				$title = esc_html( $post->post_title );

			}

			$html .= '<li itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">
						<span itemprop="item">
							<span itemprop="name">'. $title .'</span>
						</span>
						<meta itemprop="position" content="2" />
					</li>';

		}

		$html = '<li itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">
					<a itemscope itemtype="https://schema.org/Thing" itemprop="item" href="'. esc_url( home_url( '/' ) ) .'">
						<span itemprop="name">'. esc_html_x( 'Home', 'breadcrumb', 'sanfrancisco' ) .'</span>
					</a>
					<meta itemprop="position" content="1" />
				</li>'. $html;

		$html = '<ol itemscope itemtype="https://schema.org/BreadcrumbList" class="breadcrumb">'. $html .'</ol>';

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_breadcrumb_output', $html );
		}

		return apply_filters( 'sanfrancisco_breadcrumb_output', $html );

	}
}

if ( ! function_exists( 'sanfrancisco_documentation_link' ) ) {
	/**
	 * Get documentation link.
	 *
	 * @since   1.0.0
	 * @param   string $anchor
	 * @param   string $link_text
	 * @param   bool $echo
	 * @return  string
	 */
	function sanfrancisco_documentation_link( $anchor, $link_text, $echo = true ) {

		$url = SANFRANCISCO_DOCS_URI;
		if ( $anchor ) {
			$url .= '#'. $anchor;
		}

		$html = '<a href="'. esc_url( $url ) .'" target="_blank">'. esc_html( $link_text ) .'</a>';

		if ( $echo ) {
			echo apply_filters( 'sanfrancisco_documentation_link_output', $html );
		}

		return apply_filters( 'sanfrancisco_documentation_link_output', $html );

	}
}

if ( ! function_exists( 'sanfrancisco_mega_menu' ) ) {
	/**
	 * Output mega menu.
	 *
	 * @since   1.0.0
	 * @param   int    $item_id   menu item id (category for mega menu items)
	 * @param   string $item_type
	 */
	function sanfrancisco_mega_menu( $item_id = 0, $item_type = 'menu_item' ) {

		$mm_posts = array();
		if ( $item_type == 'trending_item' ) {
			$mm_posts = sanfrancisco_get_trending_posts();
		}

		if ( empty( $mm_posts ) ) {
			return;
		}

		echo '<div class="sf-mega-menu">';

		sanfrancisco_load_template( array(
			'template' => 'section',
			'prefix'   => 'loop',
			'args'     => sanfrancisco_get_mega_menu_template_args( $mm_posts )
		) );

		echo '</div>';

	}
}

if ( ! function_exists( 'sanfrancisco_ad' ) ) {
	/**
	 * Output the ad in the theme-specific way.
	 *
	 * @since   1.0.0
	 * @param   mixed $ad_id
	 * @param   string $align
	 * @param   string $location
	 * @param   string $position
	 * @param   string $breakpoint_id
	 */
	function sanfrancisco_ad( $ad_id, $align = 'center', $location = 'none', $position = '', $breakpoint_id = '' ) {

		if ( ! class_exists( 'Agg_Flex_Ads' ) ) {
			return;
		}

		$ad_obj = new Agg_Flex_Ads_Ad( $ad_id );

		if ( ! $ad_obj->is_ad ) {
			return;
		}

		$ad_output_args = array(
			'id'         => $ad_id,
			'align'      => $align,
			'wrap_class' => array(),
			'wrap_attr'  => array()
		);

		if ( ! empty( $breakpoint_id ) ) {
			$ad_output_args['visibility'] = array( $breakpoint_id );
		}

		if ( isset( $position ) ) {
			$ad_output_args['wrap_class'][] = 'sf-ad-' . $location . ( empty( $position ) ? '' : '-' . $position );
		}

		if ( $location == 'background' ) {

			$ad_output_args['wrap_attr']['id'] = $ad_obj->attr_id;
			$ad_output_args['inner_tag'] = 'a';
			$ad_output_args['inner_attr'] = array(
				'href'   => $ad_obj->options( 'background_ad_url' ),
				'target' => $ad_obj->options( 'background_ad_link_target' )
			);

		}

		if ( in_array( $location, array( 'header', 'footer', 'fixed' ) ) ) {
			$ad_output_args['wrap_class'][] = 'container';
		}

		if ( is_numeric( $ad_obj->options( $location . '_ad_opening_delay' ) ) ) {
			$ad_output_args['wrap_attr']['data-opening-delay'] = $ad_obj->options( $location . '_ad_opening_delay' );
		}

		if ( $ad_obj->options( $location . '_ad_has_countdown' ) == 'true' && is_numeric( $ad_obj->options( $location . '_ad_countdown' ) ) ) {

			$ad_output_args['wrap_attr']['data-countdown'] = absint( $ad_obj->options( $location . '_ad_countdown' ) );

			if ( preg_match( '#^(top|bottom)_(right|left)$#', $ad_obj->options( $location . '_ad_countdown_position' ) ) ) {
				$ad_output_args['wrap_attr']['data-countdown-position'] = $ad_obj->options( $location . '_ad_countdown_position' );
			}

		}

		if ( $location == 'splash' || $ad_obj->options( $location . '_ad_has_close_btn' ) == 'true' ) {

			if ( preg_match( '#^(top|bottom)_(right|left)$#', $ad_obj->options( $location . '_ad_close_btn_position' ) ) ) {
				$ad_output_args['wrap_attr']['data-close-position'] = $ad_obj->options( $location . '_ad_close_btn_position' );
			}

			$ad_close_btn_text = $ad_obj->options( $location . '_ad_close_btn_text' );

			if ( ! empty( $ad_close_btn_text ) ) {
				$ad_output_args['wrap_attr']['data-close-text'] = $ad_obj->options( $location . '_ad_close_btn_text' );
			}

		}

		echo agg_flex_ads_get_ad( $ad_output_args );

	}
}

if ( ! function_exists( 'sanfrancisco_get_options_css_selectors' ) ) {
	/**
	 * Get selector equivalents of option keys.
	 *
	 * @since   1.2.1
	 * @return  array
	 */
	function sanfrancisco_get_options_css_selectors() {

		$options_css_selectors = array(
			'body'                       => array(
				'selector'    => 'body',
				'breakpoints' => array(
					array(
						'min'        => 768,
						'properties' => array(
							'font-size'
						)
					)
				)
			),
			'selection'                  => array(
				'children' => array(
					'text' => array(
						'selector' => array(
							'::selection',
							'::-moz-selection'
						)
					)
				)
			),
			'section_wrapper'            => array(
				'selector' => '.sf-sections-wrapper'
			),
			'boxed_layout'               => array(
				'selector' => '.sf-section:not(#sf-section-mega_menu) > .container'
			),
			'headings'                   => array(
				'selector' => array(
					'h1',
					'h2',
					'h3',
					'h4',
					'h5',
					'h6'
				)
			),
			'widgets_title'              => array(
				'selector' => '.sidebar-widget .widget-title .title'
			),
			'fancy_title'                => array(
				'children' => array(
					'box' => array(
						'selector' => array(
							'.archive .sf-fancy-title',
							'.search .sf-fancy-title'
						)
					),
					'title' => array(
						'selector' => array(
							'.archive .sf-fancy-title .sf-title',
							'.search .sf-fancy-title .sf-title'
						)
					),
					'links_regular' => array(
						'selector' => array(
							'.archive .sf-fancy-title .breadcrumb a',
							'.search .sf-fancy-title .breadcrumb a'
						)
					),
					'links_hover' => array(
						'selector' => array(
							'.archive .sf-fancy-title .breadcrumb a:hover',
							'.search .sf-fancy-title .breadcrumb a:hover'
						)
					),
					'links_active' => array(
						'selector' => array(
							'.archive .sf-fancy-title .breadcrumb li > span',
							'.search .sf-fancy-title .breadcrumb li > span',
							'.archive .sf-fancy-title .breadcrumb a:active',
							'.search .sf-fancy-title .breadcrumb a:active',
							'.archive .sf-fancy-title .breadcrumb a:focus',
							'.search .sf-fancy-title .breadcrumb a:focus'
						)
					)
				)
			),
			'archive_title'              => array(
				'selector'    => array(
					'.archive .sf-fancy-title .sf-title',
					'.search .sf-fancy-title .sf-title'
				),
				'breakpoints' => array(
					array(
						'min'        => 768,
						'properties' => array(
							'font-size'
						)
					)
				)
			),
			'breadcrumb'                 => array(
				'selector'    => '.sf-fancy-title .breadcrumb',
				'breakpoints' => array(
					array(
						'min'        => 768,
						'properties' => array(
							'font-size'
						)
					)
				)
			),
			'header_logo'                => array(
				'selector' => '.navbar-brand .sf-header-logo',
				'children' => array(
					'regular' => array(
						'selector' => '.navbar-brand .sf-header-logo'
					),
					'hover'   => array(
						'selector' => '.navbar-brand .sf-header-logo:hover'
					),
					'active'  => array(
						'selector' => array(
							'.navbar-brand .sf-header-logo:active',
							'.navbar-brand .sf-header-logo:focus'
						)
					)
				)
			),
			'header_logo_mobile'         => array(
				'selector' => '.navbar-brand .sf-header-logo-mobile'
			),
			'footer'                     => array(
				'selector' => '.sf-footer'
			),
			'footer_lines'               => array(
				'selector' => array(
					'.sf-footer-brand-box hr',
					'.sf-footer-bottom > hr',
					'.sf-footer .sidebar-widget .widget-title:before',
					'.sf-footer .sidebar-widget .widget-title:after'
				)
			),
			'footer_logo'                => array(
				'children' => array(
					'regular' => array(
						'selector' => '.sf-footer-brand .sf-footer-logo'
					),
					'hover'   => array(
						'selector' => '.sf-footer-brand .sf-footer-logo:hover'
					),
					'active'  => array(
						'selector' => array(
							'.sf-footer-brand .sf-footer-logo:active',
							'.sf-footer-brand .sf-footer-logo:focus'
						)
					)
				)
			),
			'footer_logo_mobile'         => array(
				'selector' => '.sf-footer-brand .sf-footer-logo-mobile'
			),
			'footer_branding_desc'       => array(
				'selector' => '.sf-footer-brand-text'
			),
			'footer_copyright_text'      => array(
				'selector' => '.sf-footer-copyright-text'
			),
			'footer_nav'                 => array(
				'children' => array(
					'regular' => array(
						'selector' => '.sf-footer-nav li a'
					),
					'hover'   => array(
						'selector' => '.sf-footer-nav li a:hover'
					),
					'active'  => array(
						'selector' => array(
							'.sf-footer-nav li a:active',
							'.sf-footer-nav li a:focus'
						)
					)
				)
			),
			'header'                     => array(
				'selector' => '.sf-nav-wrapper'
			),
			'header_line'                => array(
				'selector' => '.navbar-header'
			),
			'main_menu'                  => array(
				'selector' => '.nav-menu > li a'
			),
			'main_menu_trending_item'    => array(
				'selector' => '.navbar-right > .sf-item-trending > a',
				'children' => array(
					'regular' => array(
						'selector' => '.navbar-nav.navbar-right > .sf-item-trending > a'
					),
					'hover'   => array(
						'selector' => '.navbar-nav.navbar-right > .sf-item-trending > a:hover'
					),
					'active'  => array(
						'selector' => array(
							'.navbar-nav.navbar-right > .sf-item-trending > a:active',
							'.navbar-nav.navbar-right > .sf-item-trending > a:focus',
							'.navbar-nav.navbar-right > .sf-item-trending.active > a',
							'.navbar-nav.navbar-right > .sf-item-trending.current-menu-item > a'
						)
					)
				)
			),
			'main_menu_top'              => array(
				'selector' => '.navbar-nav > li > a',
				'children' => array(
					'regular' => array(
						'selector' => '.navbar-nav > li > a'
					),
					'hover'   => array(
						'selector' => '.navbar-nav > li > a:hover'
					),
					'active'  => array(
						'selector' => array(
							'.navbar-nav > li > a:active',
							'.navbar-nav > li > a:focus',
							'.navbar-nav > li.active > a',
							'.navbar-nav > li.current-menu-item > a'
						)
					)
				)
			),
			'main_menu_top_current_item' => array(
				'selector' => array(
					'.navbar-nav > li.active > a',
					'.navbar-nav > li.current-menu-item > a'
				)
			),
			'main_menu_sub_box'          => array(
				'selector'    => array(
					'.navbar-nav > li .sub-menu'
				),
				'breakpoints' => array(
					array(
						'min' => 992,
					)
				)
			),
			'main_menu_sub_arrow'        => array(
				'selector'    => array(
					'.navbar-nav > li > .sub-menu > li.menu-item-has-children > a:after',
					'.navbar-nav > li.sf-sm-on > a:after'
				),
				'breakpoints' => array(
					array(
						'min' => 992,
					)
				)
			),
			'main_menu_sub'              => array(
				'selector'    => '.navbar-nav > li .sub-menu li a',
				'children'    => array(
					'regular' => array(
						'selector' => '.navbar-nav > li .sub-menu li a'
					),
					'hover'   => array(
						'selector' => '.navbar-nav > li .sub-menu li a:hover'
					),
					'active'  => array(
						'selector' => array(
							'.navbar-nav > li .sub-menu li a:active',
							'.navbar-nav > li .sub-menu li a:focus'
						)
					)
				),
				'breakpoints' => array(
					array(
						'min' => 992,
					)
				)
			),
			'main_menu_sub_current_item' => array(
				'selector'    => array(
					'.navbar-nav > li > .sub-menu li.active > a',
					'.navbar-nav > li > .sub-menu li.current-menu-item > a'
				),
				'breakpoints' => array(
					array(
						'min' => 992,
					)
				)
			),
			'search_icon'                => array(
				'children' => array(
					'regular' => array(
						'selector' => '.sf-item-search >.sf-btn-search'
					),
					'hover'   => array(
						'selector' => '.sf-item-search >.sf-btn-search:hover'
					),
					'active'  => array(
						'selector' => array(
							'.sf-item-search >.sf-btn-search:active',
							'.sf-item-search >.sf-btn-search:focus'
						)
					)
				)
			),
			'singular_primary'           => array(
				'children' => array(
					'box' => array(
						'selector' => array(
							'.sf-entry-singular'
						)
					),
					'regular' => array(
						'selector' => array(
							'.sf-entry-singular',
							'.sf-comment-form .form-control',
							'.sf-related-entries .sf-entry .sf-entry-title a',
							'.sf-related-entries .sf-entry .sf-entry-author a',
							'.sf-related-entries .sf-entry .sf-entry-meta',
							'.sf-related-entries .sf-entry .sf-entry-tags .sf-label',
							'.sf-entry-singular > .sf-entry-header .breadcrumb a'
						)
					),
					'hover' => array(
						'selector' => array(
							'.sf-related-entries .sf-entry .sf-entry-title a:hover',
							'.sf-related-entries .sf-entry .sf-entry-author a:hover',
							'.sf-comment-metadata a:hover',
							'.sf-entry-singular > .sf-entry-header .sf-entry-meta a:hover',
							'.sf-entry-singular > .sf-entry-header .breadcrumb a:hover'
						)
					),
					'active' => array(
						'selector' => array(
							'.sf-related-entries .sf-entry .sf-entry-title a:active',
							'.sf-related-entries .sf-entry .sf-entry-title a:focus',
							'.sf-related-entries .sf-entry .sf-entry-author a:active',
							'.sf-related-entries .sf-entry .sf-entry-author a:focus',
							'.sf-comment-metadata a:active',
							'.sf-comment-metadata a:focus',
							'.sf-entry-singular > .sf-entry-header .sf-entry-meta a:active',
							'.sf-entry-singular > .sf-entry-header .sf-entry-meta a:focus',
							'.sf-entry-singular > .sf-entry-header .breadcrumb a:active',
							'.sf-entry-singular > .sf-entry-header .breadcrumb a:focus'
						)
					)
				)
			),
			'singular_secondary'         => array(
				'children' => array(
					'box' => array(
						'selector' => array(
							'.sf-entry-content .sf-pagination',
							'.sf-entry-tags-box',
							'.sf-entry-content blockquote:not(.twitter-tweet):not(.instagram-media)'
						)
					),
					'regular' => array(
						'selector' => array(
							'.sf-entry-content .sf-pagination a',
							'.sf-entry-singular .sf-entry-flag-trending',
							'.sf-entry-singular > .sf-entry-header .sf-entry-subtitle',
							'.sf-entry-singular .mashsb-count',
							'.sf-entry-singular .mashsb-sharetext',
							'.sf-entry-content .agggif-caption',
							'.sf-entry-singular .sf-author-box .sf-author-bio',
							'.sf-entry-singular .sf-author-box .sf-author-social li a',
							'.sf-entry-singular .sf-entry-tags .sf-tag a',
							'.sf-related-entries .sf-entry .sf-entry-excerpt',
							'.sf-comment-content p',
							'.sf-comment-form .comment-notes',
							'.sf-comment-form .comment-notes a',
							'.sf-comment-reply a',
							'.sf-comment-respond .comment-reply-title a',
							'.sf-entry-content blockquote:not(.twitter-tweet):not(.instagram-media):before',
							'.sf-comment-form .form-control::-moz-placeholder',
							'.sf-comment-form .form-control::-webkit-input-placeholder',
							'.sf-comment-form .form-control:-ms-input-placeholder'
						)
					),
					'hover' => array(
						'selector' => array(
							'.sf-entry-content .sf-pagination a:hover',
							'.sf-entry-singular .sf-entry-tags .sf-tag a:hover',
							'.sf-comment-reply a:hover',
							'.sf-comment-respond .comment-reply-title a:hover',
							'.sf-entry-content .sf-entry-tags .sf-tag a:hover',
							'.sf-entry-singular sf-author-box .sf-author-social li a:hover',
							'.sf-comment-form .comment-notes a:hover'
						)
					),
					'active' => array(
						'selector' => array(
							'.sf-entry-content .sf-pagination a:active',
							'.sf-entry-content .sf-pagination a:focus',
							'.sf-entry-singular .sf-entry-tags .sf-tag a:active',
							'.sf-entry-singular .sf-entry-tags .sf-tag a:focus',
							'.sf-comment-reply a:active',
							'.sf-comment-reply a:focus',
							'.sf-comment-respond .comment-reply-title a:active',
							'.sf-comment-respond .comment-reply-title a:focus',
							'.sf-entry-content .sf-entry-tags .sf-tag a:active',
							'.sf-entry-content .sf-entry-tags .sf-tag a:focus',
							'.sf-entry-singular .sf-author-box .sf-author-social li a:active',
							'.sf-entry-singular .sf-author-box .sf-author-social li a:focus',
							'.sf-comment-form .comment-notes a:active',
							'.sf-comment-form .comment-notes a:focus'
						)
					)
				)
			),
			'singular_lines'             => array(
				'selector' => array(
					'.sf-entry-singular .sf-author-box',
					'.sf-entry-singular .sf-comment-list',
					'.sf-comment-form .form-control',
					'.sf-entry-singular .sf-module-title.sf-has-line:before',
					'.sf-entry-singular .sf-module-title.sf-has-line:after'
				)
			),
			'singular_category_button'   => array(
				'children' => array(
					'regular' => array(
						'selector' => '.sf-entry-singular > .sf-entry-header .sf-entry-meta .sf-btn'
					),
					'hover' => array(
						'selector' => array(
							'.sf-entry-singular > .sf-entry-header .sf-entry-meta .sf-btn:focus',
							'.sf-entry-singular > .sf-entry-header .sf-entry-meta .sf-btn:hover'
						)
					)
				)
			),
			'singular_comment_button'    => array(
				'children' => array(
					'regular' => array(
						'selector' => '.sf-comment-form .form-submit .submit'
					),
					'hover' => array(
						'selector' => array(
							'.sf-comment-form .form-submit .submit:focus',
							'.sf-comment-form .form-submit .submit:hover'
						)
					)
				)
			),
			'singular_title'             => array(
				'selector'    => '.sf-section .sf-entry-singular > .sf-entry-header .sf-entry-title',
				'breakpoints' => array(
					array(
						'min'        => 1200,
						'properties' => array(
							'font-size'
						)
					)
				)
			),
			'singular_subtitle'          => array(
				'selector'    => '.sf-entry-singular > .sf-entry-header .sf-entry-subtitle',
				'breakpoints' => array(
					array(
						'min'        => 1200,
						'properties' => array(
							'font-size'
						)
					)
				)
			),
			'singular_content'           => array(
				'selector'    => '.sf-entry-content',
				'children' => array(
					'text' => array(
						'selector' => '.sf-entry-content'
					),
					'headings' => array(
						'selector' => array(
							'.sf-entry-content h1',
							'.sf-entry-content h2',
							'.sf-entry-content h3',
							'.sf-entry-content h4',
							'.sf-entry-content h5',
							'.sf-entry-content h6'
						)
					),
					'links_regular' => array(
						'selector' => '.sf-entry-content a'
					),
					'links_hover' => array(
						'selector' => '.sf-entry-content a:hover'
					),
					'links_active' => array(
						'selector' => array(
							'.sf-entry-content a:active',
							'.sf-entry-content a:focus'
						)
					)
				),
				'breakpoints' => array(
					array(
						'min'        => 768,
						'properties' => array(
							'font-size'
						)
					)
				)
			),
			'mm_color'                   => array(
				'children' => array(
					'box' => array(
						'selector' => '.navbar-nav > li .sf-mega-menu',
					),
					'primary_regular' => array(
						'selector' => array(
							'.sf-mega-menu .sf-entry-title a',
							'.sf-mega-menu .sf-entry-meta'
						)
					),
					'primary_hover' => array(
						'selector' => '.sf-entry-title a:hover'
					),
					'primary_active' => array(
						'selector' => array(
							'.sf-entry-title a:active',
							'.sf-entry-title a:focus'
						)
					),
					'secondary_regular' => array(
						'selector' => array(
							'.sf-mega-menu .sf-entry-excerpt'
						)
					)
				)
			),
			'trending_icon'              => array(
				'selector' => '.sf-entry-flag-trending'
			),
			'sidebar_widgets_primary'    => array(
				'children' => array(
					'box'     => array(
						'selector' => array(
							'.sf-sidebar .sidebar-widget',
							'#wp-calendar thead th',
							'#wp-calendar tbody td'
						)
					),
					'regular' => array(
						'selector' => array(
							'.sf-sidebar .sidebar-widget',
							'.sf-sidebar .sidebar-widget a',
							'.sf-about-me .author-name',
							'.sidebar-widget .search-form .input-group-addon',
							'.sidebar-widget .search-form .form-control',
							'.mashsb-widget-link',
							'#wp-calendar > caption',
							'#wp-calendar thead th',
							'.sidebar-widget .search-form .form-control::-moz-placeholder',
							'.sidebar-widget .search-form .form-control::-webkit-input-placeholder',
							'.sidebar-widget .search-form .form-control:-ms-input-placeholder'
						)
					),
					'hover'   => array(
						'selector' => array(
							'.sf-sidebar .sidebar-widget a:hover',
							'.mashsb-widget-link:hover'
						)
					),
					'active'  => array(
						'selector' => array(
							'.sf-sidebar .sidebar-widget a:active',
							'.sf-sidebar .sidebar-widget a:focus',
							'.mashsb-widget-link:active',
							'.mashsb-widget-link:focus'
						)
					)
				)
			),
			'sidebar_widgets_secondary'  => array(
				'children' => array(
					'box'     => array(
						'selector' => array(
							'.widget_nav_menu .menu',
							'.widget_recent_comments > ul',
							'.widget_pages > ul',
							'.widget_archive > ul',
							'.widget_meta > ul',
							'.widget_categories > ul',
							'.widget_recent_entries > ul',
							'.widget_rss > ul',
							'.sf-about-me',
							'.sf-instagram-images',
							'.sidebar-widget .search-form .input-group-addon',
							'.sidebar-widget .search-form .form-control',
							'#calendar_wrap',
							'#wp-calendar thead th',
							'#wp-calendar > caption',
							'#wp-calendar tbody td#today',
							'.widget_tag_cloud .tagcloud'
						)
					),
					'regular' => array(
						'selector' => array(
							'.sf-sidebar .widget_recent_comments',
							'.sf-about-me .author-title',
							'.sf-about-me .author-social-icons li a',
							'.sf-about-me .author-description',
							'.sf-sidebar .sf-social-icons li a',
							'.widget_recent_entries > ul li .post-date',
							'.widget_rss > ul li .rss-date',
							'.widget_rss > ul li .rssSummary',
							'.widget_rss > ul li cite',
							'.mashsb-excerpt',
							'.mashicon-share',
							'#wp-calendar tbody td',
							'#wp-calendar tfoot td#next a',
							'#wp-calendar tfoot td#prev a'
						)
					),
					'hover'   => array(
						'selector' => array(
							'.sf-about-me .author-social-icons li a:hover',
							'.sf-sidebar .sf-social-icons li a:hover',
							'#wp-calendar tfoot td#next a:hover',
							'#wp-calendar tfoot td#prev a:hover'
						)
					),
					'active'  => array(
						'selector' => array(
							'.sf-about-me .author-social-icons li a:active',
							'.sf-about-me .author-social-icons li a:focus',
							'.sf-sidebar .sf-social-icons li a:active',
							'.sf-sidebar .sf-social-icons li a:focus',
							'#wp-calendar tfoot td#next a:active',
							'#wp-calendar tfoot td#next a:focus',
							'#wp-calendar tfoot td#prev a:active',
							'#wp-calendar tfoot td#prev a:focus'
						)
					)
				)
			),
			'sidebar_widgets_button'     => array(
				'children' => array(
					'regular' => array(
						'selector' => '.widget_tag_cloud .tagcloud a'
					),
					'hover'   => array(
						'selector' => '.widget_tag_cloud .tagcloud a:hover'
					)
				)
			),
			'sidebar_widgets_title'      => array(
				'selector' => '.sf-sidebar .sidebar-widget .widget-title .title'
			),
			'sidebar_widgets_title_line' => array(
				'selector' => array(
					'.sidebar-widget .widget-title:before',
					'.sidebar-widget .widget-title:after'
				)
			),
			'footer_widgets_title'       => array(
				'selector' => '.sf-footer .sidebar-widget .widget-title .title'
			),
			'footer_widgets_text'        => array(
				'selector' => '.sf-footer .sidebar-widget'
			),
			'footer_widgets_links'       => array(
				'selector' => '.sf-footer .sidebar-widget a',
				'children' => array(
					'regular' => array(
						'selector' => '.sf-footer .sidebar-widget a'
					),
					'hover'   => array(
						'selector' => '.sf-footer .sidebar-widget a:hover'
					),
					'active'  => array(
						'selector' => array(
							'.sf-footer .sidebar-widget a:active',
							'.sf-footer .sidebar-widget a:focus'
						)
					)
				)
			),
			'gp_image'                   => array(
				'children' => array(
					'overlay'        => array(
						'selector'    => '.sf-entry-featured-media > a:before',
						'breakpoints' => array(
							array(
								'min' => 768
							)
						)
					),
					'overlay_hover'  => array(
						'selector'    => '.sf-entry-featured-media > a:hover:before',
						'breakpoints' => array(
							array(
								'min' => 768
							)
						)
					),
					'overlay_mobile' => array(
						'selector'    => '.sf-entry-featured-media > a:before',
						'breakpoints' => array(
							array(
								'max' => 767
							)
						)
					)
				)
			),
			'gp_box'                     => array(
				'selector' => '.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry'
			),
			'gp_primary'                 => array(
				'children' => array(
					'regular' => array(
						'selector' => array(
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-title a',
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-author a',
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-meta',
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-tags .sf-label',
						)
					),
					'hover'   => array(
						'selector' => array(
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry:hover .sf-entry-title a',
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry .sf-entry-author a:hover',
						)
					),
					'active'  => array(
						'selector' => array(
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-title a:active',
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-title a:focus',
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-author a:active',
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-author a:focus'
						)
					)
				)
			),
			'gp_secondary'               => array(
				'children' => array(
					'regular' => array(
						'selector' => array(
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-excerpt',
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-tags .sf-tag'
						)
					),
					'hover'   => array(
						'selector' => '.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-tags .sf-tag a:hover'
					),
					'active'  => array(
						'selector' => array(
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-tags .sf-tag a:active',
							'.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-entry-tags .sf-tag a:focus'
						)
					)
				)
			),
			'gp_button'                  => array(
				'children' => array(
					'regular' => array(
						'selector' => '.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-btn'
					),
					'hover'   => array(
						'selector' => '.sf-sections-wrapper .sf-module-grid-posts:not(.sf-module-widget):not(.sf-related-entries) .sf-btn:hover'
					)
				)
			),
			'fp'                         => array(
				'children' => array(
					'overlay'              => array(
						'selector'    => '.sf-entry-body',
						'breakpoints' => array(
							array(
								'min'        => 992,
								'properties' => array(
									'background-color',
									'color'
								)
							)
						)
					),
					'overlay_hover'        => array(
						'selector'    => array(
							'.sf-entry-body:hover',
							'.sf-entry:hover .sf-entry-body',
							//@todo: fp-hover These two lines added to fix featured posts nav button hover issue.
							/*'.sf-entries-featured.sf-carousel:hover .sf-entry-body',
							'.sf-entries-featured.sf-carousel:hover .sf-entry .sf-entry-body'*/
						),
						'breakpoints' => array(
							array(
								'min'        => 992,
								'properties' => array(
									'background-color',
									'color'
								)
							)
						)
					),
					'overlay_mobile'       => array(
						'selector'    => '.sf-entry-body',
						'breakpoints' => array(
							array(
								'max'        => 991,
								'properties' => array(
									'background-color',
									'color'
								)
							)
						)
					),
					'category_line'        => array(
						'selector'    => array(
							'.sf-entry-category:before',
							'.sf-entry-category:after'
						),
						'breakpoints' => array(
							array(
								'min'        => 992,
								'properties' => array(
									'background-color',
									'color'
								)
							)
						)
					),
					'category_line_mobile' => array(
						'selector'    => array(
							'.sf-entry-category:before',
							'.sf-entry-category:after'
						),
						'breakpoints' => array(
							array(
								'max'        => 991,
								'properties' => array(
									'background-color',
									'color'
								)
							)
						)
					),
					'carousel_buttons'     => array(
						'selector' => '.sf-carousel-controls .sf-control'
					)
				)
			),
			'module'                     => array(
				'children' => array(
					'title_wrap' => array(
						'selector' => '.sf-module-title'
					),
					'title'     => array(
						'selector' => '.sf-module-title .sf-title'
					),
					'title_bar' => array(
						'selector' => array(
							'.sf-module-title.sf-has-line:before',
							'.sf-module-title.sf-has-line:after'
						)
					)
				)
			),
			'fixed_bg_ad'                => array(
				'selector' => '.aggads-inner'
			),
			'fixed_bg'                   => array(
				'selector' => '.sf-bg-fixed'
			),
			'author_box'                 => array(
				'children' => array(
					'box' => array(
						'selector' => '.author #sf-module-author-box .sf-author-box'
					),
					'title' => array(
						'selector' => '.author .sf-author-box .sf-author-title'
					),
					'bio' => array(
						'selector' => '.author .sf-author-box .sf-author-bio'
					),
					'links_regular' => array(
						'selector' => '.author .sf-author-box .sf-author-social a'
					),
					'links_hover' => array(
						'selector' => '.author .sf-author-box .sf-author-social a:hover'
					),
					'links_active' => array(
						'selector' => array(
							'.author .sf-author-box .sf-author-social a:active',
							'.author .sf-author-box .sf-author-social a:focus'
						)
					)
				)
			),
			'pagination'                 => array(
				'children' => array(
					'box' => array(
						'selector' => '.sf-pagination'
					),
					'links_regular' => array(
						'selector' => array(
							'.sf-pagination > .pagination .page-numbers',
							'.sf-pagination > .pagination .page-numbers:after'
						)
					),
					'links_hover' => array(
						'selector' => '.sf-pagination > .pagination a.page-numbers:hover'
					),
					'links_active' => array(
						'selector' => '.sf-pagination > .pagination .page-numbers.current'
					)
				)
			)
		);
		
		return $options_css_selectors;

	}
}

if ( ! function_exists( 'sanfrancisco_get_options_css' ) ) {
	/**
	 * Get options of css elements extracted from theme options.
	 *
	 * @since   1.0.0
	 * @return  array
	 */
	function sanfrancisco_get_options_css( $option_affix = '' ) {

		global $post;

		if ( sanfrancisco_is_singular() && ! isset( $post->ID ) ) {
			return array();
		}

		if ( ! empty( $option_affix ) ) {
			$option_affix = '__' . str_replace( '__', '', $option_affix );
		}

		$opts = array(
			'background'
		);

		// Get global options
		if ( ! empty( $option_affix ) ) {

			$opts = array_merge( $opts, array(
				'body_font',
				'selection_color',
				'headings_font',
				'widgets_title_font',
				'archive_title_font',
				'breadcrumb_font',
				'header_logo_font',
				'header_logo_color',
				'header_logo_width',
				'header_logo_padding_y',
				'header_logo_mobile_font',
				'header_logo_mobile_color',
				'header_logo_mobile_width',
				'header_logo_mobile_padding_y',
				'footer_background_color',
				'footer_lines_background_color',
				'footer_logo_font',
				'footer_logo_color',
				'footer_logo_mobile_font',
				'footer_logo_mobile_color',
				'footer_branding_desc_font',
				'footer_branding_desc_color',
				'footer_copyright_text_color',
				'footer_nav_color',
				'header_background',
				'header_line_border_color',
				'main_menu_font',
				'main_menu_trending_item_font',
				'main_menu_trending_item_color',
				'main_menu_top_text_transform',
				'main_menu_top_color',
				'main_menu_top_current_item_font_weight',
				'main_menu_sub_box_background_color',
				'main_menu_sub_arrow_color',
				'main_menu_sub_font_size',
				'main_menu_sub_text_transform',
				'main_menu_sub_color',
				'main_menu_sub_current_item_font_weight',
				'search_icon_color',
				'trending_icon_background_color',
				'sidebar_widgets_primary_color',
				'sidebar_widgets_secondary_color',
				'sidebar_widgets_button_color',
				'sidebar_widgets_title_color',
				'sidebar_widgets_title_line_background_color',
				'footer_widgets_title_color',
				'footer_widgets_text_color',
				'footer_widgets_links_color',
				'gp_box_background_color',
				'gp_primary_color',
				'gp_secondary_color',
				'gp_button_color',
				'gp_image_color',
				'pagination_color',
				'mm_color',
				'fancy_title_color'
			) );

		}

		if ( is_singular() ) {

			$opts = wp_parse_args( array(
				'singular_primary_color',
				'singular_secondary_color',
				'singular_lines_border_color',
				'singular_category_button_color',
				'singular_comment_button_color',
				'singular_title_font',
				'singular_subtitle_font',
				'singular_content_font',
				'singular_content_color',
				'singular_content_line_height',
			), $opts );

		} else if ( is_author() ) {

			$opts[] = 'author_box_color';

		}

		foreach ( $opts as $k => $v ) {
			$opts[ $k ] .= $option_affix;
		}

		$result = sanfrancisco_get_options( $opts );

		if ( is_array( $result ) ) {
			foreach ( $result as $k => $v ) {
				unset( $result[$k] );
				$k = preg_replace( '#' . $option_affix . '$#', '', $k );
				$result[$k] = $v;
			}
		}

		if ( sanfrancisco_get_option( 'boxed_layout' . $option_affix ) == 'true' ) {
			$result = wp_parse_args( sanfrancisco_get_options( array(
				'boxed_layout_background_color' . $option_affix
			) ), $result );
		}

		$page_sidebars = array();
		$posts_widget_id = 'sanfrancisco-posts';
		$saved_posts_widgets = get_option( 'widget_' . $posts_widget_id );
		$sidebars_widgets = wp_get_sidebars_widgets();

		if ( sanfrancisco_get_option( 'display_footer_sidebars' . $option_affix ) == 'true' ) {
			foreach ( $sidebars_widgets as $k => $v ) {
				if ( false !== strpos( $k, 'footer-sidebar' ) ) {
					$page_sidebars[] = $k;
				}
			}
		}

		// Get featured post style options in page builder modules
		if ( sanfrancisco_is_singular() && get_page_template_slug( $post->ID ) == 'page-sections.php' ) {

			global $post;

			$sections = get_post_meta( $post->ID, sanfrancisco_get_pb_id(), true );

			if ( is_array( $sections ) ) {
				foreach ( $sections as $section ) {

					if ( $section['settings']['layout'] == 'ws' ) {
						$page_sidebars[] = $section['settings']['sidebar'];
					}

					if ( array_key_exists( 'modules', $section ) && is_array( $section['modules'] ) ) {
						foreach ( $section['modules'] as $module ) {

							$id_prefix = 'sf-module-'. $module['id'];

							if ( in_array( $module['layout'], array( 'grid_posts', 'advertisement' ) ) ) {

								if ( array_key_exists( 'title_font', $module ) ) {
									$result['module'][ 'title_font@'. $id_prefix ] = $module['title_font'];
								}

								if ( array_key_exists( 'title_wrap_margin_bottom', $module ) ) {
									$result['module'][ 'title_wrap_margin_bottom@'. $id_prefix ] = $module['title_wrap_margin_bottom'];
								}

								if ( array_key_exists( 'title_color', $module ) && is_array( $module['title_color'] ) ) {

									if ( array_key_exists( 'text_color', $module['title_color'] ) ) {
										$result['module'][ 'title_color@'. $id_prefix ] = $module['title_color']['text_color'];
									}

									if ( array_key_exists( 'bar_background_color', $module['title_color'] ) ) {
										$result['module'][ 'title_bar_background_color@'. $id_prefix ] = $module['title_color']['bar_background_color'];
									}

								}

								if ( array_key_exists( 'title_bar_height', $module ) ) {
									$result['module'][ 'title_bar_height@'. $id_prefix ] = $module['title_bar_height'];
								}

							} else if ( $module['layout'] == 'featured_posts' ) {

								if ( array_key_exists( 'fp_color', $module ) && is_array( $module['fp_color'] ) ) {

									if ( array_key_exists( 'overlay_mobile_background_color', $module['fp_color'] ) ) {
										$result['fp'][ 'overlay_mobile_background_color@'. $id_prefix ] = $module['fp_color']['overlay_mobile_background_color'];
									}

									if ( array_key_exists( 'overlay_mobile_color', $module['fp_color'] ) ) {
										$result['fp'][ 'overlay_mobile_color@'. $id_prefix ] = $module['fp_color']['overlay_mobile_color'];
										$result['fp'][ 'category_line_mobile_background_color@'. $id_prefix ] = $module['fp_color']['overlay_mobile_color'];
									}

									if ( array_key_exists( 'overlay_background_color', $module['fp_color'] ) ) {
										$result['fp'][ 'overlay_background_color@'. $id_prefix ] = $module['fp_color']['overlay_background_color'];
									}

									if ( array_key_exists( 'overlay_color', $module['fp_color'] ) ) {
										$result['fp'][ 'overlay_color@'. $id_prefix ] = $module['fp_color']['overlay_color'];
										$result['fp'][ 'category_line_background_color@'. $id_prefix ] = $module['fp_color']['overlay_mobile_color'];
									}

									if ( array_key_exists( 'overlay_hover_background_color', $module['fp_color'] ) ) {
										$result['fp'][ 'overlay_hover_background_color@'. $id_prefix ] = $module['fp_color']['overlay_hover_background_color'];
									}

									if ( array_key_exists( 'overlay_hover_color', $module['fp_color'] ) ) {
										$result['fp'][ 'overlay_hover_color@'. $id_prefix ] = $module['fp_color']['overlay_hover_color'];
									}

									if ( array_key_exists( 'carousel_buttons_color', $module['fp_color'] ) ) {
										$result['fp'][ 'carousel_buttons_color@'. $id_prefix ] = $module['fp_color']['carousel_buttons_color'];
									}

									if ( array_key_exists( 'carousel_buttons_border_color', $module['fp_color'] ) ) {
										$result['fp'][ 'carousel_buttons_border_color@'. $id_prefix ] = $module['fp_color']['carousel_buttons_border_color'];
									}

								}

							}

						}
					}

				}
			}

		} else {
			$page_sidebars[] = sanfrancisco_get_option( 'sidebar' . $option_affix );
		}

		// Get featured posts style options in widgets
		foreach ( $sidebars_widgets as $sidebar_id => $widgets ) {
			if ( in_array( $sidebar_id, $page_sidebars ) && is_array( $widgets ) ) {
				foreach ( $widgets as $widget_id ) {
					if ( preg_match( '#^' . $posts_widget_id . '-([0-9]+)$#', $widget_id, $match ) ) {

						$widget_index = end( $match );

						if ( array_key_exists( $widget_index, $saved_posts_widgets ) && array_key_exists( 'posts_layout', $saved_posts_widgets[ $widget_index ] ) && preg_match( '#^style_([a-z]{1})$#', $saved_posts_widgets[ $widget_index ]['posts_layout'] ) ) {

							if ( array_key_exists( 'fp_color', $saved_posts_widgets[ $widget_index ] ) && is_array( $saved_posts_widgets[ $widget_index ]['fp_color'] ) ) {

								if ( array_key_exists( 'overlay_mobile_background_color', $saved_posts_widgets[ $widget_index ]['fp_color'] ) ) {
									$result['fp']['overlay_mobile_background_color@'. $widget_id ] = $saved_posts_widgets[ $widget_index ]['fp_color']['overlay_mobile_background_color'];
								}

								if ( array_key_exists( 'overlay_mobile_color', $saved_posts_widgets[ $widget_index ]['fp_color'] ) ) {
									$result['fp']['overlay_mobile_color@'. $widget_id ] = $saved_posts_widgets[ $widget_index ]['fp_color']['overlay_mobile_color'];
									$result['fp']['category_line_mobile_background_color@'. $widget_id ] = $saved_posts_widgets[ $widget_index ]['fp_color']['overlay_mobile_color'];
								}

								if ( array_key_exists( 'overlay_background_color', $saved_posts_widgets[ $widget_index ]['fp_color'] ) ) {
									$result['fp']['overlay_background_color@'. $widget_id ] = $saved_posts_widgets[ $widget_index ]['fp_color']['overlay_background_color'];
								}

								if ( array_key_exists( 'overlay_color', $saved_posts_widgets[ $widget_index ]['fp_color'] ) ) {
									$result['fp']['overlay_color@'. $widget_id ] = $saved_posts_widgets[ $widget_index ]['fp_color']['overlay_color'];
									$result['fp']['category_line_background_color@'. $widget_id ] = $saved_posts_widgets[ $widget_index ]['fp_color']['overlay_mobile_color'];
								}

								if ( array_key_exists( 'overlay_hover_background_color', $saved_posts_widgets[ $widget_index ]['fp_color'] ) ) {
									$result['fp']['overlay_hover_background_color@'. $widget_id ] = $saved_posts_widgets[ $widget_index ]['fp_color']['overlay_hover_background_color'];
								}

								if ( array_key_exists( 'overlay_hover_color', $saved_posts_widgets[ $widget_index ]['fp_color'] ) ) {
									$result['fp']['overlay_hover_color@'. $widget_id ] = $saved_posts_widgets[ $widget_index ]['fp_color']['overlay_hover_color'];
								}

								if ( array_key_exists( 'carousel_buttons_color', $saved_posts_widgets[ $widget_index ]['fp_color'] ) ) {
									$result['fp']['carousel_buttons_color@'. $widget_id ] = $saved_posts_widgets[ $widget_index ]['fp_color']['carousel_buttons_color'];
								}

								if ( array_key_exists( 'carousel_buttons_border_color', $saved_posts_widgets[ $widget_index ]['fp_color'] ) ) {
									$result['fp']['carousel_buttons_border_color@'. $widget_id ] = $saved_posts_widgets[ $widget_index ]['fp_color']['carousel_buttons_border_color'];
								}

							}

						}

					}
				}
			}
		}

		// Get ad style options
		if ( class_exists( 'Agg_Flex_Ads' ) ) {

			$ad_meta = sanfrancisco_get_option( 'advertisement' . $option_affix );

			if ( is_array( $ad_meta ) && array_key_exists( 'background', $ad_meta ) && is_array( $ad_meta[ 'background' ] ) ) {

				foreach ( $ad_meta['background'] as $breakpoint_id => $ad_id ) {

					if ( ! is_numeric( $ad_id ) ) {
						continue;
					}

					$ad_obj = new Agg_Flex_Ads_Ad( $ad_id );
					$ad_options = $ad_obj->options();

					if ( is_array( $ad_options ) && array_key_exists( 'background_ad_attachment_id', $ad_options ) ) {

						$result['fixed_bg_ad_background@' . $ad_obj->attr_id ] = array(
							'background_image'      => array_key_exists( 'background_ad_attachment_id', $ad_options ) ? $ad_options['background_ad_attachment_id'] : '',
							'background_attachment' => array_key_exists( 'background_ad_image_attachment', $ad_options ) ? $ad_options['background_ad_image_attachment'] : 'fixed',
							'background_repeat'     => array_key_exists( 'background_ad_image_repeat', $ad_options ) ? str_replace( '_', '-', $ad_options['background_ad_image_repeat'] ) : 'no-repeat',
							'background_position'   => 'top_center'
						);

						if ( array_key_exists( 'background_ad_content_top_spacing', $ad_options ) ) {
							$result[ 'section_wrapper_padding_top#' . $breakpoint_id ] = $ad_options['background_ad_content_top_spacing'];
						}

					}

				}

			}

		}

		foreach ( $result as $k => $v ) {

			$k_noaffix = preg_replace( '#' . $option_affix . '$#', '', $k );

			$kp = Agg_Option_Styles::parse_option_key( $k_noaffix );

			if ( $kp['key'] == 'background' ) {

				unset( $result[ $k ] );

				$prefix = ( ! empty( $v['background_image'] ) && $v['background_attachment'] == 'fixed' ) ? 'fixed_bg' : 'section_wrapper';

				$result[ $prefix . '_' . $k ] = $v;

			}

		}

		return $result;

	}
}

if ( ! function_exists( 'sanfrancisco_gp_posts' ) ) {
	/**
	 * (Ajax) Output posts for grid layout
	 *
	 * @since   1.3.1
	 */
	function sanfrancisco_gp_posts() {

		sanfrancisco_load_template( array(
			'template' => 'grid_posts',
			'args'     => $_POST
		) );

		exit;

	}
}
add_action( 'wp_ajax_sanfrancisco_gp_posts', 'sanfrancisco_gp_posts' );
add_action( 'wp_ajax_nopriv_sanfrancisco_gp_posts', 'sanfrancisco_gp_posts' );