<?php

if ( ! function_exists( 'sanfrancisco_is_page' ) ) {
	/**
	 * Alternative to is_page. This function checks current page on admin panel.
	 *
	 * @since   1.2.0
	 * @return  bool
	 */
	function sanfrancisco_is_page() {

		global $current_screen;

		if ( is_null( $current_screen ) ) {
			return is_page();
		}

		return ( $current_screen->post_type == 'page' );

	}
}

if ( ! function_exists( 'sanfrancisco_is_single' ) ) {
	/**
	 * Alternative to is_single. This function checks current page on admin panel.
	 *
	 * @since   1.2.0
	 * @return  bool
	 */
	function sanfrancisco_is_single() {

		global $current_screen;
	
		if ( is_null( $current_screen ) ) {
			return is_page();
		}
	
		return ( $current_screen->post_type == 'page' );
	
	}
}

if ( ! function_exists( 'sanfrancisco_is_singular' ) ) {
	/**
	 * Alternative to is_singular. This function checks current page on admin panel.
	 *
	 * @since   1.2.0
	 * @return  bool
	 */
	function sanfrancisco_is_singular() {

		global $current_screen;
	
		if ( is_null( $current_screen ) ) {
			return is_singular();
		}
	
		return ( array_key_exists( $current_screen->post_type, sanfrancisco_get_post_types() ) );
	
	}
}

if ( ! function_exists( 'sanfrancisco_get_locale' ) ) {
	/**
	 * Get the current locale.
	 *
	 * @since   1.0.0
	 * @return  string $locale
	 */
	function sanfrancisco_get_locale() {

		$locale = get_locale();

		if( preg_match( '#^[a-z]{2}\-[A-Z]{2}$#', $locale ) ) {
			$locale = str_replace( '-', '_', $locale );
		} else if ( preg_match( '#^[a-z]{2}$#', $locale ) ) {
			$locale .= '_'. mb_strtoupper( $locale, 'UTF-8' );
		}

		if ( empty( $locale ) ) {
			$locale = 'en_US';
		}

		return $locale;

	}
}

if ( ! function_exists( 'sanfrancisco_locate_template_uri' ) ) {
	/**
	 * Retrieve the name of the highest priority template file uri that exists.
	 *
	 * Searches in the STYLESHEETPATH before TEMPLATEPATH
	 * so that themes which inherit from a parent theme can just overload one file.
	 *
	 * @since 2.7.0
	 *
	 * @param string|array $template_names Template file(s) to search for, in order.
	 * @return string The template file uri if one is located.
	 */
	function sanfrancisco_locate_template_uri( $template_names ) {

		$located = '';

		foreach ( (array) $template_names as $template_name ) {
			if ( !$template_name )
				continue;
			if ( file_exists(STYLESHEETPATH . '/' . $template_name)) {
				$located = get_stylesheet_directory_uri() . '/' . $template_name;
				break;
			} elseif ( file_exists(TEMPLATEPATH . '/' . $template_name) ) {
				$located = get_template_directory_uri() . '/' . $template_name;
				break;
			}
		}

		return $located;

	}
}

if ( ! function_exists( 'sanfrancisco_helper_classes' ) ) {
	/**
	 * Require helper classes.
	 *
	 * @since   1.0.0
	 */
	function sanfrancisco_helper_classes() {

		if ( ! class_exists( 'Hybrid_Media_Grabber' ) ) {
			require_once trailingslashit( get_template_directory() ) . 'library/assets/lib/class-hybrid-media-grabber.php';
		}

		if ( ! class_exists( 'Agg_Option_Styles' ) ) {
			require_once trailingslashit( get_template_directory() ) . 'library/assets/lib/class-agg-option-styles.php';
		}

	}
}
add_action( 'init', 'sanfrancisco_helper_classes' );

if ( ! function_exists( 'array_column' ) ) {
	/**
	 * array_column function compatibility for old php version (< 5.5.0).
	 *
	 * @since   1.0.0
	 * @param   array $array      an array (record set) from which to pull a column of values
	 * @param   string $column    the column of values to return
	 * @return  array $array      returns an array of values representing a single column from the input array.
	 */
	function array_column( $array, $column ) {

		$result = array();

		if ( ! is_array( $array ) ) {
			$array = array();
		}

		foreach ( $array as $k => $v ) {
			if ( $k == $column ) {
				$result[] = $v;
			}
		}

		return $result;

	}
}

if ( ! function_exists( 'sanfrancisco_get_post_types' ) ) {
	/**
	 * Get all post types.
	 *
	 * @since   1.0.0
	 * @return  array $post_types
	 */
	function sanfrancisco_get_post_types() {

		/* get registered post types */
		$post_types = get_post_types( '', false );

		/* these post types will not be retrieved */
		$forbidden_post_types = array( 'attachment', 'revision', 'nav_menu_item' );

		foreach ( $post_types as $k => $v ) {
			if ( in_array( $k, $forbidden_post_types ) || ! $v->public ) {
				unset( $post_types[ $k ] );
				continue;
			}
			$post_types[ $k ] = array(
				'label' => $v->labels->name,
				'value' => $k
			);
		}

		return $post_types;

	}
}

if ( ! function_exists( 'sanfrancisco_get_image_sizes' ) ) {
	/**
	 * Get all available image sizes generated by theme.
	 *
	 * @since   1.0.0
	 * @param   string $size
	 * @return  array
	 */
	function sanfrancisco_get_image_sizes( $size = '' ) {

		$image_sizes = array(
			'post-thumbnail' => array(
				'width'  => 228,
				'height' => 171,
				'ratio'  => 75
			),
			'sf-160-120'     => array(
				'width'  => 160,
				'height' => 120,
				'ratio'  => 75
			),
			'sf-320-160'     => array(
				'width'  => 320,
				'height' => 160,
				'ratio'  => 50
			),
			'sf-320-240'     => array(
				'width'  => 320,
				'height' => 240,
				'ratio'  => 75
			),
			'sf-380-285'     => array(
				'width'  => 380,
				'height' => 285,
				'ratio'  => 75
			),
			'sf-380-570'     => array(
				'width'  => 380,
				'height' => 570,
				'ratio'  => 150
			),
			'sf-570-300'     => array(
				'width'  => 570,
				'height' => 285,
				'ratio'  => 50,
			),
			'sf-570-570'     => array(
				'width'  => 570,
				'height' => 570,
				'ratio'  => 100
			),
			'sf-700-525'     => array(
				'width'  => 700,
				'height' => 525,
				'ratio'  => 75
			),
			'sf-710-355'     => array(
				'width'  => 710,
				'height' => 355,
				'ratio'  => 50
			),
			'sf-760-570'     => array(
				'width'  => 760,
				'height' => 570,
				'ratio'  => 75
			),
			'sf-760-1140'    => array(
				'width'  => 760,
				'height' => 1140,
				'ratio'  => 150
			),
			'sf-750-655'     => array(
				'width'  => 750,
				'height' => 660,
				'ratio'  => 88
			),
			'sf-1140-570'    => array(
				'width'  => 1140,
				'height' => 570,
				'ratio'  => 50
			),
			'sf-1140-855'    => array(
				'width'  => 1140,
				'height' => 855,
				'ratio'  => 75
			),
			'sf-1140-1140'   => array(
				'width'  => 1140,
				'height' => 1140,
				'ratio'  => 100
			)
		);

		return ( ! is_string( $size ) || ! array_key_exists( $size, $image_sizes ) ) ? $image_sizes : $image_sizes[ $size ];

	}
}

if ( ! function_exists( 'sanfrancisco_load_template' ) ) {
	/**
	 * Load template file.
	 *
	 * @since   1.0.0
	 * @param   array $sanfrancisco_args            template args
	 * @return  array $sanfrancisco_template_vars   template vars
	 */
	function sanfrancisco_load_template( $sanfrancisco_args ) {

		if ( ! array_key_exists( 'template', $sanfrancisco_args ) || ! is_string( $sanfrancisco_args['template'] ) || empty( $sanfrancisco_args['template'] ) ) {
			return array();
		}

		$template_name = $sanfrancisco_args['template'];

		if ( ! array_key_exists( 'prefix', $sanfrancisco_args ) || ! is_string( $sanfrancisco_args['prefix'] ) || empty( $sanfrancisco_args['prefix'] ) ) {
			$sanfrancisco_args['prefix'] = 'tpl';
		}

		$prefix = $sanfrancisco_args['prefix'];

		if ( ! array_key_exists( 'args', $sanfrancisco_args ) || ! is_array( $sanfrancisco_args['args'] ) ) {
			$sanfrancisco_args['args'] = array();
		}

		/** @noinspection PhpUnusedLocalVariableInspection */
		$sanfrancisco_args = $sanfrancisco_args[ 'args'];

		/** @noinspection PhpIncludeInspection */
		include( trailingslashit( get_template_directory() ) . 'library/templates/'. sanitize_key( $prefix ) .'-'. sanitize_key( $template_name ) .'.php' );

		if ( isset( $sanfrancisco_template_vars ) ) {
			return $sanfrancisco_template_vars;
		}

		return array();

	}
}

if ( ! function_exists( 'sanfrancisco_get_option' ) ) {
	/**
	 * Get option.
	 *
	 * @since   1.0.0
	 * @param   string $field           option field
	 * @param   bool $ignore_override   if true ignore parent override and return most specific option
	 * @param   array $args             option args
	 * @return  mixed
	 */
	function sanfrancisco_get_option( $field, $ignore_override = false, $args = array() ) {

		$prefix = 'sanfrancisco_';
		$filter_key = 'sanfrancisco_get_option';
		$inheritance = array(
			'advertisement_inheritance' => array(
				'advertisement'
			),
			'background_inheritance' => array(
				'background'
			)
		);

		if ( ! is_array( $args ) ) {
			$args = array();
		}

		if ( array_key_exists( 'post_id', $args ) ) {
			$post = get_post( $args['post_id'] );
		} else if ( sanfrancisco_is_singular() ) { // We used "sanfrancisco_is_singular" in order for us to check if we are editing a singular post as well.
			global $post;
		}

		$field = preg_replace( '#^'. $prefix .'#is', '', $field );
		$to_settings = get_option( sanfrancisco_get_to_id() .'_settings', array() );
		$std_values = array();

		foreach ( $to_settings as $k => $v ) {
			if ( ! in_array( $v['type'], array( 'section', 'tab' ) ) ) {
				$std_values[ $v['id'] ] = array_key_exists( 'std', $v ) ? $v['std'] : '';
			}
		}

		$field_raw = $field;
		$affix_pattern = '#__(home|tag|author|search|archive|category|single|page|woocommerce|bbpress|buddypress)$#is';
		$std_pattern = '#__(std)$#is';

		if ( $has_affix = preg_match( $affix_pattern, $field, $match_affix ) ) {
			$field_raw = preg_replace( $affix_pattern, '', $field );
		}

		if ( preg_match( $std_pattern, $field ) ) {

			$field = $field_raw = preg_replace( $std_pattern, '', $field );

			if ( preg_match( $affix_pattern, $field ) ) {
				$field_raw = preg_replace( $affix_pattern, '', $field );
			}

			if ( array_key_exists( $field, $std_values ) ) {
				return apply_filters( $filter_key, $std_values[ $field ], $field_raw );
			} else if ( array_key_exists( $field_raw, $std_values ) ) {
				return apply_filters( $filter_key, $std_values[ $field_raw ], $field_raw );
			} else {
				return apply_filters( $filter_key, '', $field_raw );
			}

		}

		foreach ( $inheritance as $k => $v ) {
			if ( in_array( $field_raw, $v ) ) {
				$parent_field = $k;
				break;
			}
		}

		if ( ( is_category() || is_tag() ) && ! $has_affix ) {

			$term = get_queried_object();
			$affix = is_category() ? '__category' : '__tag';

			if ( ! $ignore_override && get_term_meta( $term->term_id, $prefix . 'ignore_override', true ) != 'true' && sanfrancisco_get_option( 'override'. $affix ) == 'true' ) {
				return sanfrancisco_get_option( $field_raw . $affix, $ignore_override, $args );
			}

			if ( isset( $parent_field ) ) {
				$value = get_term_meta( $term->term_id, $prefix . $parent_field, true );
				if ( $value == 'inherit' ) {
					return sanfrancisco_get_option( $field_raw . $affix );
				}
			}

			$value = get_term_meta( $term->term_id, $prefix . $field_raw );

			if ( ! empty( $value ) && $value[0] != 'inherit' ) {
				return apply_filters( $filter_key, $value[0], $field_raw );
			}

			return sanfrancisco_get_option( $field_raw . $affix );

		} else if ( isset( $post ) && $post && ! $has_affix ) {

			$affix = ( $post->post_type == 'page' ) ? '__page' : '__single';

			if ( ! $ignore_override && get_post_meta( $post->ID, $prefix . 'ignore_override', true ) != 'true' && sanfrancisco_get_option( 'override'. $affix ) == 'true' ) {
				return sanfrancisco_get_option( $field_raw . $affix, $ignore_override, $args );
			}

			if ( isset( $parent_field ) ) {
				$value = get_post_meta( $post->ID, $prefix . $parent_field, true );
				if ( $value == 'inherit' ) {
					return sanfrancisco_get_option( $field_raw . $affix );
				}
			}

			$value = get_post_meta( $post->ID, $prefix . $field_raw );

			if ( ! empty( $value ) && $value[0] != 'inherit' ) {
				return apply_filters( $filter_key, $value[0], $field_raw );
			}

			return sanfrancisco_get_option( $field_raw . $affix );

		} else {

			if ( ! $has_affix ) {

				$has_affix = true;

				if ( is_home() ) {
					$affix = '__home';
				} else if ( is_tag() ) {
					$affix = '__tag';
				} else if ( is_author() ) {
					$affix = '__author';
				} else if ( is_search() ) {
					$affix = '__search';
				} else if ( is_archive() ) {
					$affix = '__archive';
				} else {
					$has_affix = false;
				}

				if ( isset( $affix ) ) {
					$field = $field_raw . $affix;
				}

			} else {

				$affix = '__'. end( $match_affix );

			}

			/* get saved theme options */
			$options = get_option( sanfrancisco_get_to_id(), array() );

			if ( ! is_array( $options ) ) {
				return sanfrancisco_get_option( $field .'__std', false );
			}

			if ( isset( $parent_field ) && isset( $affix ) && $has_affix ) {
				$parent_field .= $affix;
				$value = array_key_exists( $parent_field, $options ) ? $options[ $parent_field ] : '';
				if ( $value == 'inherit' ) {
					return array_key_exists( $field_raw, $options ) ? apply_filters( $filter_key, $options[ $field_raw ], $field_raw ) : sanfrancisco_get_option( $field_raw .'__std' );
				}
			}

			if ( array_key_exists( $field, $options ) ) {
				return apply_filters( $filter_key, $options[ $field ], $field_raw );
			} else if ( $has_affix && array_key_exists( $field, $std_values ) ) {
				return apply_filters( $filter_key, $std_values[ $field ], $field_raw );
			} else if ( array_key_exists( $field_raw, $options ) ) {
				return apply_filters( $filter_key, $options[ $field_raw ], $field_raw );
			} else {
				return sanfrancisco_get_option( $field .'__std' );
			}

		}

	}
}

if ( ! function_exists( 'sanfrancisco_get_options' ) ) {
	/**
	 * Get options.
	 *
	 * @since   1.0.0
	 * @param   array $fields           option fields as an array
	 * @param   bool $ignore_override   if true ignore parent override and return most specific option
	 * @param   array $args             option args
	 * @return  mixed
	 */
	function sanfrancisco_get_options( $fields = array(), $ignore_override = false, $args = array() ) {

		if ( ! is_array( $fields ) ) {
			$fields = array();
		}

		$options = array();
		foreach ( $fields as $k ) {
			$options[ $k ] = sanfrancisco_get_option( $k, $ignore_override, $args );
		}

		return $options;

	}
}

if ( ! function_exists( 'sanfrancisco_get_to_id' ) ) {
	/**
	 * Get theme options option key.
	 *
	 * @since   1.0.0
	 * @return  string
	 */
	function sanfrancisco_get_to_id() {

		return apply_filters( 'sanfrancisco_theme_options_id', 'sanfrancisco_to' );

	}
}

if ( ! function_exists( 'sanfrancisco_get_pb_id' ) ) {
	/**
	 * Get page builder options meta key.
	 *
	 * @since   1.0.0
	 * @return  string
	 */
	function sanfrancisco_get_pb_id() {

		return apply_filters( 'sanfrancisco_page_builder_id', 'sanfrancisco_pb' );

	}
}

if ( ! function_exists( 'sanfrancisco_views_meta_key' ) ) {
	/**
	 * Get post views meta key.
	 *
	 * @since   1.0.0
	 * @return  string
	 */
	function sanfrancisco_views_meta_key() {

		return 'sanfrancisco_views';

	}
}

if ( ! function_exists( 'sanfrancisco_likes_meta_key' ) ) {
	/**
	 * Get post likes meta key.
	 *
	 * @since   1.0.0
	 * @return  string
	 */
	function sanfrancisco_likes_meta_key() {

		return 'sanfrancisco_likes';

	}
}

if ( ! function_exists( 'sanfrancisco_unlikes_meta_key' ) ) {
	/**
	 * Get post unlikes meta key.
	 *
	 * @since   1.0.0
	 * @return  string
	 */
	function sanfrancisco_unlikes_meta_key() {

		return 'sanfrancisco_unlikes';

	}
}

if ( ! function_exists( 'sanfrancisco_get_post_query_args' ) ) {
	/**
	 * Get post query args by given args.
	 *
	 * @since   1.0.0
	 * @param   array $args
	 *  == Available args ==
	 *      * ignore_sticky_posts (bool)
	 *      * posts_per_page (int)
	 *      * posts_offset (int)
	 *      * post_types (array)
	 *      * post_ids (array)
	 *      * excluded_post_ids (array)
	 *      * post_taxonomies (array) [Format: {TERM-ID}_{TAX-NAME}]
	 *      * post_tags (string|array)
	 *      * order (string) [asc|desc]
	 *      * orderby (string) [date|views|votes|comments|random|post_ids]
	 * @return  array $_args
	 */
	function sanfrancisco_get_post_query_args( $args ) {

		if ( ! is_array( $args ) ) {
			$args = array();
		}

		$_args = array(
			'post_status'   => 'publish',
			'no_found_rows' => false
		);

		if ( array_key_exists( 'ignore_sticky_posts', $args ) ) {
			$_args['ignore_sticky_posts'] = $args['ignore_sticky_posts'];
		}

		if ( array_key_exists( 'paged', $args ) && is_numeric( $args['paged'] ) ) {
			$_args['paged'] = $args['paged'];
		} elseif ( get_query_var( 'paged' ) ) {
			$_args['paged'] = get_query_var( 'paged' );
		} elseif ( get_query_var( 'page' ) ) {
			$_args['paged'] = get_query_var( 'page' );
		} else {
			$_args['paged'] = 1;
		}

		if ( array_key_exists( 'posts_per_page', $args ) ) {

			$_args['posts_per_page'] = $args['posts_per_page'];

			if ( array_key_exists( 'posts_offset', $args ) ) {
				$_args['offset'] = ( ( $_args['paged'] - 1 ) * $_args['posts_per_page'] ) + $args['posts_offset'];
			}

		}

		if ( array_key_exists( 'post_types', $args ) && ! empty( $args['post_types'] ) && is_array( $args['post_types'] ) ) {
			$_args['post_type'] = $args['post_types'];
		}

		if ( array_key_exists( 'post_ids', $args ) && ! empty( $args['post_ids'] ) ) {

			$post_ids = is_array( $args['post_ids'] ) ? $args['post_ids'] : explode( ',', $args['post_ids'] );

			$_args['post__in'] = array_filter( array_map( function( $v ) {
				return trim( $v );
			}, $post_ids ), function( $v ) {
				return is_numeric( $v );
			} );

			if ( empty( $_args['post__in'] ) ) {
				unset( $_args['post__in'] );
			}

		}

		if ( array_key_exists( 'excluded_post_ids', $args ) && ! empty( $args['excluded_post_ids'] ) ) {

			$excluded_post_ids = is_array( $args['excluded_post_ids'] ) ? $args['excluded_post_ids'] : explode( ',', $args['excluded_post_ids'] );

			$_args['post__not_in'] = array_filter( array_map( function( $v ) {
				return trim( $v );
			}, $excluded_post_ids ), function( $v ) {
				return is_numeric( $v );
			} );

			if ( empty( $_args['post__not_in'] ) ) {
				unset( $_args['post__not_in'] );
			}

		}

		if ( array_key_exists( 'post_taxonomies', $args ) && ! empty( $args['post_taxonomies'] ) && is_array( $args['post_taxonomies'] ) ) {

			$_args['tax_query'] = array(
				'relation' => 'OR'
			);

			$post_taxonomies = array();
			foreach ( $args['post_taxonomies'] as $k => $v ) {
				if ( preg_match( '#^([0-9]+)_(.*)$#', $v, $match ) ) {
					if ( ! array_key_exists( $match[2], $post_taxonomies ) ) {
						$post_taxonomies[ $match[2] ] = array();
					}
					$post_taxonomies[ $match[2] ][] = $match[1];
				}
			}

			foreach ( $post_taxonomies as $k => $v ) {
				$_args['tax_query'][] = array(
					'taxonomy' => $k,
					'field' => 'term_id',
					'terms' => $v
				);
			}
		}

		if ( array_key_exists( 'post_tags', $args ) && ! empty( $args['post_tags'] ) ) {

			if ( is_string( $args['post_tags'] ) ) {

				$args['post_tags'] = array_map( function( $v ) {
					return trim( $v );
				}, explode( ',', $args['post_tags'] ) );

			}

			if ( is_array( $args['post_tags'] ) ) {

				$post_tags = array_filter( $args['post_tags'], function( $v ) {
					return ( ! empty( $v ) );
				} );

				if ( ! empty( $post_tags ) ) {

					$tax_query_tags = array(
						array(
							'taxonomy' => 'post_tag',
							'field'    => 'slug',
							'terms'    => $post_tags
						)
					);

					if ( ! array_key_exists( 'post_type', $_args ) || in_array( 'product', $_args['post_type'] ) ) {

						$tax_query_tags[] = array(
							'taxonomy' => 'product_tag',
							'field'    => 'slug',
							'terms'    => $post_tags
						);

					}

					if ( count( $tax_query_tags ) > 1 ) {
						$tax_query_tags['relation'] = 'OR';
					}

					$_args['tax_query'] = ( ! array_key_exists( 'tax_query', $_args ) ) ? $tax_query_tags : array(
						'relation' => 'AND',
						$_args['tax_query'],
						$tax_query_tags
					);

				}

			}

		}

		if ( array_key_exists( 'order', $args ) ) {
			$_args['order'] = ( $args['order'] == 'asc' ) ? 'ASC' : 'DESC';
		}

		if ( array_key_exists( 'orderby', $args ) ) {
			switch ( $args['orderby'] ) {
				case 'views':
					$_args['orderby'] = 'meta_value_num';
					$_args['meta_key'] = sanfrancisco_views_meta_key();
					break;
				case 'votes':
					$_args['orderby'] = 'meta_value_num';
					$_args['meta_key'] = sanfrancisco_votes_meta_key();
					break;
				case 'comments':
					$_args['orderby'] = 'comment_count';
					break;
				case 'random':
					$_args['orderby'] = 'rand';
					break;
				case 'post_ids':
					$_args['orderby'] = 'post__in';
					break;
				default:
					$_args['orderby'] = 'date';
			}
		}

		return $_args;

	}
}

if ( ! function_exists( 'sanfrancisco_get_archive_template_args' ) ) {
	/**
	 * Get archive template args.
	 *
	 * @since   1.0.0
	 * @return  array $args
	 */
	function sanfrancisco_get_archive_template_args() {

		$args = array();

		$opts = sanfrancisco_get_options( array(
			'layout',
			'sidebar',
			'sidebar_position',
			'display_title',
			'display_breadcrumb',
			'display_author_info',
			'posts_layout',
			'pagination_type',
			'meta_data',
			'excerpt_length',
			'display_tags',
			'tag_limit'
		) );

		$id = 'archives';
		if ( is_home() ) {
			$id = 'home';
		} else if ( is_category() ) {
			$id = 'category';
		} else if ( is_search() ) {
			$id = 'search';
		} else if ( is_author() ) {
			$id = 'author';
		} else if ( is_tag() ) {
			$id = 'tag';
		}

		$args[ $id ] = array(
			'id' => $id,
			'settings' => array(
				'layout'           => $opts['layout'],
				'sidebar'          => $opts['sidebar'],
				'sidebar_position' => $opts['sidebar_position']
			),
			'modules' => array()
		);

		if ( $id != 'home' ) {

			if ( in_array( 'true', array( $opts['display_title'], $opts['display_breadcrumb'] ), true ) ) {

				$module_id = $id .'-title';

				$p = array(
					'id' => $module_id,
					'layout' => 'fancy_title',
					'heading' => get_the_archive_title(),
					'display_title' => $opts['display_title'],
					'display_breadcrumb' => $opts['display_breadcrumb'],
					'class_name' => ''
				);

				if ( $opts['display_title'] != 'false' ) {
					$p['class_name'] .= ' sf-has-title';
				}

				if ( $opts['display_breadcrumb'] != 'false' ) {
					$p['class_name'] .= ' sf-has-breadcrumb';
				}

				$p['class_name'] = trim( $p['class_name'] );
				$args[ $id ]['modules'][ $module_id ] = $p;

			}

		}

		if ( $id == 'author' && $opts['display_author_info'] == 'true' ) {

			$module_id = $id .'-box';
			$author = get_queried_object();

			$p = array(
				'id' => $module_id,
				'layout' => 'author_box',
				'author_id' => $author->ID
			);

			$args[ $id ]['modules'][ $module_id ] = $p;

		}

		$args[ $id ]['modules'][ $id ] = array(
			'id'                 => $id,
			'layout'             => 'grid_posts',
			'wp_query'           => true,
			'gp_posts_layout'    => $opts['posts_layout'],
			'gp_pagination_type' => $opts['pagination_type'],
			'meta_data'          => $opts['meta_data'],
			'excerpt_length'     => $opts['excerpt_length'],
			'display_tags'       => $opts['display_tags'],
			'tag_limit'          => $opts['tag_limit']
		);

		return $args;

	}
}

if ( ! function_exists( 'sanfrancisco_get_singular_template_args' ) ) {
	/**
	 * Get singular template args.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @return  mixed
	 */
	function sanfrancisco_get_singular_template_args( $post = null ) {

		$args = array();

		$post = get_post( $post );
		if ( ! $post ) {
			return false;
		}

		if ( ! isset( $post ) ) {
			global $post;
		}

		$id = $post->post_type;

		$args[ $id ] = array(
			'id' => $id,
			'settings' => sanfrancisco_get_options( array(
				'layout',
				'sidebar',
				'sidebar_position'
			), false, array(
				'post_id' => $post->ID
			) ),
			'modules' => array()
		);

		$args[ $id ]['modules'][ $id ] = array_merge( array(
			'id' => $id,
			'layout' => 'singular'
		), sanfrancisco_get_options( array(
			'meta_data',
			'page_ad',
			'tag_limit',
			'display_title',
			'display_title_subline',
			'display_featured_image',
			'featured_image_style',
			'display_credit_line',
			'display_content',
			'display_tags',
			'display_like_button',
			'display_related_posts',
			'display_pagination',
			'display_author_info',
			'display_comments',
			'display_page_ad',
			'dropcap'
		), false, array(
			'post_id' => $post->ID
		) ) );

		return $args;

	}
}

if ( ! function_exists( 'sanfrancisco_get_mega_menu_template_args' ) ) {
	/**
	 * Get mega menu template args.
	 *
	 * @since   1.0.0
	 * @param   array $posts
	 * @return  array $args
	 */
	function sanfrancisco_get_mega_menu_template_args( $posts = array() ) {

		$args = array();

		$opts = sanfrancisco_get_options( array(
			'trending_posts_layout',
			'trending_posts_meta_data',
			'trending_posts_excerpt_length',
			'trending_posts_display_tags',
			'trending_posts_tag_limit',
		    'trending_posts_number',
			'trending_posts_excluded_ids',
			'trending_posts_order',
			'trending_posts_orderby'
		) );

		$post_ids = array();
		foreach ( $posts as $p ) {
			$post_ids[] = $p->ID;
		}

		$id = 'mega_menu';

		$args[ $id ] = array(
			'id' => $id,
			'settings' => array(
				'layout' => 'full'
			),
			'modules' => array()
		);

		$args[ $id ]['modules'][ $id ] = array(
			'id'                 => $id,
			'layout'             => 'grid_posts',
			'gp_posts_layout'    => $opts['trending_posts_layout'],
			'gp_pagination_type' => 'none',
			'meta_data'          => $opts['trending_posts_meta_data'],
			'excerpt_length'     => $opts['trending_posts_excerpt_length'],
			'display_tags'       => $opts['trending_posts_display_tags'],
			'tag_limit'          => $opts['trending_posts_tag_limit'],
			'posts_per_page'     => $opts['trending_posts_number'],
			'post_ids'           => $post_ids,
			'excluded_post_ids'  => $opts['trending_posts_excluded_ids'],
			'order'              => $opts['trending_posts_order'],
			'orderby'            => $opts['trending_posts_orderby']
		);

		return $args;

	}
}

if ( ! function_exists( 'sanfrancisco_class' ) ) {
	/**
	 * Convert given class array into attr value.
	 *
	 * @since   1.0.0
	 * @param   array $classes
	 * @return  string
	 */
	function sanfrancisco_class( $classes = array() ) {

		if ( ! is_array( $classes ) ) {
			$classes = array();
		}

		foreach ( $classes as $k => $v ) {
			$classes[ $k ] = sanitize_html_class( $v );
		}

		return implode( ' ',  $classes );

	}
}

if ( ! function_exists( 'sanfrancisco_get_image_url' ) ) {
	/**
	 * Get attachment image url.
	 *
	 * @since   1.0.0
	 * @param   int $attachment_id
	 * @param   string $size
	 * @param   bool $echo
	 * @return  string
	 */
	function sanfrancisco_get_image_url( $attachment_id, $size = 'full', $icon = true, $echo = false ) {

		$image_url = wp_get_attachment_image_url( $attachment_id, $size, $icon );

		if ( ! $image_url ) {
			return '';
		}

		if ( $echo ) {
			echo esc_url( $image_url );
		}

		return $image_url;

	}
}

if ( ! function_exists( 'sanfrancisco_get_trending_posts' ) ) {
	/**
	 * Get trending posts.
	 *
	 * @since   1.0.0
	 * @return  array
	 */
	function sanfrancisco_get_trending_posts() {

		$transient_key = 'sanfrancisco_trending_post_ids';

		if ( ( $trending_posts = get_transient( $transient_key ) ) === false ) {

			$post_types = sanfrancisco_get_post_types();

			$trending_posts = array();

			if ( function_exists( 'stats_get_csv' ) && sanfrancisco_get_option( 'trending_auto_detect' ) != 'false' ) {

				$post_stats = stats_get_csv( 'postviews', 'days=2&limit=15' );

				if ( is_array( $post_stats ) ) {

					$page_on_front = get_option( 'page_on_front' );

					foreach ( $post_stats as $stat ) {
						if ( array_key_exists( 'post_id', $stat ) && ! empty( $stat['post_id'] ) ) {

							$p = get_post( $stat['post_id'] );

							if ( $p
								&& array_key_exists( $p->post_type, $post_types )
								&& $p->post_status == 'publish'
								&& $p->ID != $page_on_front ) {
								$trending_posts[ $p->ID ] = $p;
							}

						}
					}

				}

			}

			$query_args = array(
				'post_status'    => 'publish',
				'posts_per_page' => '-1',
				'meta_query'     => array(
					array(
						'key'     => 'sanfrancisco_is_trending',
						'value'   => 'true',
						'compare' => '=',
					)
				)
			);

			$query = new WP_Query( $query_args );

			foreach ( $query->posts as $p ) {
				$trending_posts[ $p->ID ] = $p;
			}

			set_transient($transient_key, $trending_posts, 300);

		}

		if ( ! is_array( $trending_posts ) ) {
			$trending_posts = array();
		}

		return $trending_posts;

	}
}

if ( ! function_exists( 'sanfrancisco_post_is_trending' ) ) {
	/**
	 * Check if post is trending.
	 *
	 * @since   1.1.0
	 * @param   object $post
	 * @return  bool
	 */
	function sanfrancisco_post_is_trending( $post = null ) {

		$post = get_post( $post );
		if ( ! $post ) {
			return false;
		}

		return ( array_key_exists( $post->ID, sanfrancisco_get_trending_posts() ) );

	}
}

if ( ! function_exists( 'sanfrancisco_get_related_post_ids' ) ) {
	/**
	 * Get related post ids.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @param   int  $number
	 * @return  array
	 */
	function sanfrancisco_get_related_post_ids( $post = null, $number = 3 ) {

		$post_ids = array();

		$post = get_post( $post );
		if ( ! $post ) {
			return $post_ids;
		}

		$post_taxonomies = array(
			apply_filters( 'sanfrancisco_tag_tax_names', array() ),
			apply_filters( 'sanfrancisco_category_tax_names', array() )
		);

		foreach ( $post_taxonomies as $tax_names ) {

			foreach ( $tax_names as $post_type => $tax ) {

				if ( $number > 0 ) {

					$post_terms = wp_get_post_terms( $post->ID, $tax );

					if ( is_array( $post_terms ) ) {

						$params = array(
							'ignore_sticky_posts' => true,
							'post_type'           => $post_type,
							'post_status'         => 'publish',
							'posts_per_page'      => $number,
							'post__not_in'        => array( $post->ID ),
							'tax_query'           => array(
								'relation' => 'OR'
							)
						);

						foreach ( $post_terms as $t ) {
							$params['tax_query'][] = array(
								'taxonomy' => $tax,
								'field'    => 'slug',
								'terms'    => $t->slug
							);
						}

						$query = new WP_Query( $params );

						if ( $query->have_posts() ) {
							while( $query->have_posts() ) {
								$query->the_post();
								$post_ids[] = get_the_ID();
							}
							wp_reset_postdata();
						}

						$number -= $query->post_count;

					}

				}

			}

		}

		return $post_ids;

	}
}

if ( ! function_exists( 'sanfrancisco_set_post_like' ) ) {
	/**
	 * Perform post like action.
	 *
	 * @since   1.0.0
	 */
	function sanfrancisco_set_post_like() {

		$payload = array();

		if ( ! array_key_exists( 'post_id', $_POST ) || ! array_key_exists( 'type', $_POST ) ) {
			die( json_encode( $payload ) );
		}

		$type = ( $_POST['type'] == 'unlike' ) ? 'unlike' : 'like';
		$post_id = $_POST['post_id'];
		$set_val = ( $_POST['set'] == 'true' ) ? 1 : -1;
		$ck_key = 'sanfrancisco_' . $type . 'd_posts';

		$meta_key_func = 'sanfrancisco_' . $type . 's_meta_key';
		$meta_key = $meta_key_func();

		$payload['count'] = sanfrancisco_post_likes( $type, $post_id, false );
		$ck_data = ( array_key_exists( $ck_key, $_COOKIE ) && is_string( $_COOKIE[ $ck_key ] ) ) ? json_decode( wp_unslash( $_COOKIE[ $ck_key ] ), true ) : array();

		// do nothing when
		// performing an unlike action if the current user have not liked the post before
		// or
		// performing a like action if the current user have liked the post before
		if ( ! is_array( $ck_data )
			|| ( ! in_array( $post_id, $ck_data ) && $set_val < 0 )
			|| ( in_array( $post_id, $ck_data ) && $set_val > 0 ) ) {
			die( json_encode( $payload ) );
		}

		$payload['count'] += $set_val;
		if ( $payload['count'] < 0 ) {
			$payload['count'] = 0;
		}

		if ( $set_val < 0 ) {
			// remove liked post data from cookie for the current user if performing an unlike action.
			foreach ( $ck_data as $k => $v ) {
				if ( $v == $post_id ) {
					unset( $ck_data[ $k ] );
					break;
				}
			}
		} else {
			// mark this post as liked for the current user in the cookie if performing a like action.
			$ck_data[] = $post_id;
		}

		$ck_data = array_unique( $ck_data );

		// hold post like data in the cookie throughout 30 days for the current user
		setcookie( $ck_key, json_encode( $ck_data ), time()+60*60*24*30, '/' );

		// update post like meta data
		update_post_meta( $post_id, $meta_key, $payload['count'] );

		echo json_encode( $payload );

		exit;

	}
}
add_action( 'wp_ajax_sanfrancisco_set_post_like', 'sanfrancisco_set_post_like' );
add_action( 'wp_ajax_nopriv_sanfrancisco_set_post_like', 'sanfrancisco_set_post_like' );

if ( ! function_exists( 'sanfrancisco_post_is' ) ) {
	/**
	 * Conditional check for post by user cookie.
	 *
	 * @since   1.0.0
	 * @param   object $post
	 * @return  bool
	 */
	function sanfrancisco_post_is( $type = 'liked', $post = null ) {

		global $_COOKIE;

		$post = get_post( $post );
		if ( ! $post ) {
			return false;
		}

		$ck_key = 'sanfrancisco_' . $type . '_posts';

		if ( ! array_key_exists( $ck_key, $_COOKIE ) ) {
			return false;
		}

		$ck_data = json_decode( wp_unslash( $_COOKIE[ $ck_key ] ), true );

		if ( ! is_array( $ck_data ) || ! in_array( $post->ID, $ck_data ) ) {
			return false;
		}

		return true;

	}
}

if ( ! function_exists( 'sanfrancisco_get_styles' ) ) {
	/**
	 * Get custom styles generated by theme options.
	 *
	 * @since   1.0.0
	 * @param   string $return  custom css generated by theme options of google fonts url
	 * @return  mixed
	 */
	function sanfrancisco_get_styles( $return = 'css' ) {

		$key = 'sanfrancisco_options_css';

		if ( is_category() || is_tag() || is_tax() ) {

			$term = get_queried_object();

			$affix = is_tag() ? '__tag' : '__category';

			$options_css_global = get_option( $key . $affix );
			$options_css_tax = get_term_meta( $term->term_id, $key, true );

			if ( ! is_array( $options_css_global ) ) {
				$options_css_global = sanfrancisco_get_options_css( $affix );
				update_option( $key . $affix, $options_css_global );
			}

			if( ! is_array( $options_css_tax ) ) {
				$options_css_tax = sanfrancisco_get_options_css();
				update_term_meta( $term->term_id, $key, $options_css_tax );
			}

			$options_css = ( get_term_meta( $term->term_id,  'ignore_override', true ) == 'true' || sanfrancisco_get_option( 'override'. $affix ) != 'true' ) ? array_replace_recursive( $options_css_global, $options_css_tax ) : array_replace_recursive( $options_css_tax, $options_css_global );

		} else if ( is_singular() ) {

			global $post;

			$affix = is_page() ? '__page' : '__single';

			$options_css_global = get_option( $key . $affix );
			$options_css_singular = get_post_meta( $post->ID, $key, true );

			if ( ! is_array( $options_css_global ) ) {
				$options_css_global = sanfrancisco_get_options_css( $affix );
				update_option( $key . $affix, $options_css_global );
			}

			if( ! is_array( $options_css_singular ) ) {
				$options_css_singular = sanfrancisco_get_options_css();
				update_post_meta( $post->ID, $key, $options_css_singular );
			}

			$options_css = ( get_post_meta( $post->ID,  'ignore_override', true ) == 'true' || sanfrancisco_get_option( 'override'. $affix ) != 'true' ) ? array_replace_recursive( $options_css_global, $options_css_singular ) : array_replace_recursive( $options_css_singular, $options_css_global );

		} else {

			if ( is_home() ) {
				$affix = '__home';
			} else if ( is_author() ) {
				$affix = '__author';
			} else if ( is_search() ) {
				$affix = '__search';
			} else if ( is_archive() ) {
				$affix = '__archive';
			} else {
				$affix = '__archive';
			}

			$options_css = get_option( $key . $affix );

			if( ! is_array( $options_css ) ) {
				$options_css = sanfrancisco_get_options_css( $affix );
				update_option( $key . $affix, $options_css );
			}

		}

		$option_styles = new Agg_Option_Styles( $options_css );

		if ( $return == 'css' ) {
			$result = $option_styles->get_css( sanfrancisco_get_options_css_selectors() );
		} else if ( $return == 'fonts_url' ) {
			$result = $option_styles->get_fonts_url( apply_filters( 'sanfrancisco_predefined_fonts', array() ) );
		} else {
			$result = '';
		}

		return $result;

	}
}

if ( ! function_exists( 'sanfrancisco_calculate_image_srcset' ) ) {
	/**
	 * A helper function to calculate the image sources to include in a 'srcset' attribute.
	 *
	 * @param array  $size_array    Array of width and height values in pixels (in that order).
	 * @param string $image_src     The 'src' of the image.
	 * @param array  $image_meta    The image meta data as returned by 'wp_get_attachment_metadata()'.
	 * @param int    $attachment_id Optional. The image attachment ID to pass to the filter. Default 0.
	 * @return string|bool          The 'srcset' attribute value. False on error or when only one source exists.
	 */
	 function sanfrancisco_calculate_image_srcset( $size_array, $image_src, $image_meta, $attachment_id = 0 ) {

		global $wp;
		
		if ( empty( $image_meta['sizes'] ) || ! isset( $image_meta['file'] ) || strlen( $image_meta['file'] ) < 4 ) {
			return false;
		}

		$image_sizes = $image_meta['sizes'];

		// Get the width and height of the image.
		$image_width = (int) $size_array[0];
		$image_height = (int) $size_array[1];

		// Bail early if error/no width.
		if ( $image_width < 1 ) {
			return false;
		}

		$image_basename = wp_basename( $image_meta['file'] );

		/*
		* WordPress flattens animated GIFs into one frame when generating intermediate sizes.
		* To avoid hiding animation in user content, if src is a full size GIF, a srcset attribute is not generated.
		* If src is an intermediate size GIF, the full size is excluded from srcset to keep a flattened GIF from becoming animated.
		*/
		if ( ! isset( $image_sizes['thumbnail']['mime-type'] ) || 'image/gif' !== $image_sizes['thumbnail']['mime-type'] ) {
			$image_sizes[] = array(
				'width'  => $image_meta['width'],
				'height' => $image_meta['height'],
				'file'   => $image_basename,
			);
		} elseif ( strpos( $image_src, $image_meta['file'] ) ) {
			return false;
		}

		// Retrieve the uploads sub-directory from the full size image.
		$dirname = _wp_get_attachment_relative_path( $image_meta['file'] );

		if ( $dirname ) {
			$dirname = trailingslashit( $dirname );
		}

		$upload_dir = wp_get_upload_dir();
		$image_baseurl = trailingslashit( $upload_dir['baseurl'] ) . $dirname;

		$parsed_url = parse_url( home_url( $wp->request ) );

		/*
		* If currently on HTTPS, prefer HTTPS URLs when we know they're supported by the domain
		* (which is to say, when they share the domain name of the current request).
		*/
		if ( is_ssl() && 'https' !== substr( $image_baseurl, 0, 5 ) && parse_url( $image_baseurl, PHP_URL_HOST ) === $parsed_url['host'] ) {
			$image_baseurl = set_url_scheme( $image_baseurl, 'https' );
		}

		/*
		* Images that have been edited in WordPress after being uploaded will
		* contain a unique hash. Look for that hash and use it later to filter
		* out images that are leftovers from previous versions.
		*/
		$image_edited = preg_match( '/-e[0-9]{13}/', wp_basename( $image_src ), $image_edit_hash );

		/**
		 * Filters the maximum image width to be included in a 'srcset' attribute.
		 *
		 * @since 4.4.0
		 *
		 * @param int   $max_width  The maximum image width to be included in the 'srcset'. Default '1600'.
		 * @param array $size_array Array of width and height values in pixels (in that order).
		 */
		$max_srcset_image_width = apply_filters( 'max_srcset_image_width', 1600, $size_array );

		// Array to hold URL candidates.
		$sources = array();

		/**
		 * To make sure the ID matches our image src, we will check to see if any sizes in our attachment
		 * meta match our $image_src. If no matches are found we don't return a srcset to avoid serving
		 * an incorrect image. See #35045.
		 */
		$src_matched = false;

		/*
		* Loop through available images. Only use images that are resized
		* versions of the same edit.
		*/
		foreach ( $image_sizes as $image ) {
			$is_src = false;

			// Check if image meta isn't corrupted.
			if ( ! is_array( $image ) ) {
				continue;
			}

			// If the file name is part of the `src`, we've confirmed a match.
			if ( ! $src_matched && false !== strpos( $image_src, $dirname . $image['file'] ) ) {
				$src_matched = $is_src = true;
			}

			// Filter out images that are from previous edits.
			if ( $image_edited && ! strpos( $image['file'], $image_edit_hash[0] ) ) {
				continue;
			}

			/*
			* Filters out images that are wider than '$max_srcset_image_width' unless
			* that file is in the 'src' attribute.
			*/
			if ( $max_srcset_image_width && $image['width'] > $max_srcset_image_width && ! $is_src ) {
				continue;
			}

			// If the image dimensions are within 1px of the expected size, use it.
			if ( wp_image_matches_ratio( $image_width, $image_height, $image['width'], $image['height'] ) ) {
				// Add the URL, descriptor, and value to the sources array to be returned.
				$source = array(
					'url'        => $image_baseurl . $image['file'],
					'descriptor' => 'w',
					'value'      => $image['width'],
				);

				// The 'src' image has to be the first in the 'srcset', because of a bug in iOS8. See #35030.
				if ( $is_src ) {
					$sources = array( $image['width'] => $source ) + $sources;
				} else {
					$sources[ $image['width'] ] = $source;
				}
			}
		}

		// Only return a 'srcset' value if there is more than one source.
		if ( ! $src_matched || ! is_array( $sources ) || count( $sources ) < 2 ) {
			return false;
		}

		$srcset = '';

		foreach ( $sources as $source ) {
			$srcset .= str_replace( ' ', '%20', $source['url'] ) . ' ' . $source['value'] . $source['descriptor'] . ', ';
		}

		return rtrim( $srcset, ', ' );

	 }
}

if ( ! function_exists( 'sanfrancisco_calculate_image_sizes' ) ) {
	/**
	 * Creates a 'sizes' attribute value for an image.
	 *
	 * @param array|string $size          Image size to retrieve. Accepts any valid image size, or an array
	 *                                    of width and height values in pixels (in that order). Default 'medium'.
	 * @param string       $image_src     Optional. The URL to the image file. Default null.
	 * @param array        $image_meta    Optional. The image meta data as returned by 'wp_get_attachment_metadata()'.
	 *                                    Default null.
	 * @param int          $attachment_id Optional. Image attachment ID. Either `$image_meta` or `$attachment_id`
	 *                                    is needed when using the image size name as argument for `$size`. Default 0.
	 * @return string|bool A valid source size value for use in a 'sizes' attribute or false.
	 */
	function sanfrancisco_calculate_image_sizes( $size, $image_src = null, $image_meta = null, $attachment_id = 0 ) {

		$width = 0;

		if ( is_array( $size ) ) {
			$width = absint( $size[0] );
		} elseif ( is_string( $size ) ) {
			if ( ! $image_meta && $attachment_id ) {
				$image_meta = wp_get_attachment_metadata( $attachment_id );
			}

			if ( is_array( $image_meta ) ) {
				$size_array = _wp_get_image_size_from_meta( $size, $image_meta );
				if ( $size_array ) {
					$width = absint( $size_array[0] );
				}
			}
		}

		if ( ! $width ) {
			return false;
		}

		// Setup the default 'sizes' attribute.
		$sizes = sprintf( '(max-width: %1$dpx) 100vw, %1$dpx', $width );

		return $sizes;
		
	}
}