<?php

class Agg_Option_Styles {

	/**
	 * @since   1.0.0
	 * @var     array
	 */
	public $css_properties = array(
		'margin',
		'margin_x',
		'margin_y',
		'margin_top',
		'margin_right',
		'margin_bottom',
		'margin_left',
		'padding',
		'padding_x',
		'padding_y',
		'padding_top',
		'padding_right',
		'padding_bottom',
		'padding_left',
		'width',
		'height',
		'border_color',
		'border_top_color',
		'border_right_color',
		'border_bottom_color',
		'border_left_color',
		'border_style',
		'border_top_style',
		'border_right_style',
		'border_bottom_style',
		'border_left_style',
		'border_width',
		'border_top_width',
		'border_right_width',
		'border_bottom_width',
		'border_left_width',
		'text_transform',
		'font',
		'font_weight',
		'font_size',
		'line_height',
		'background',
		'background_color',
		'color'
	);

	/**
	 * Arguments passed into the class and parsed with the defaults.
	 *
	 * @since  1.0.0
	 * @var    array
	 */
	public $options_css = array();

	/**
	 * Agg_Option_Styles constructor.
	 *
	 * @since   1.0.0
	 * @param   array $options_css
	 */
	public function __construct( $options_css = array() ) {

		$this->options_css = $options_css;

		if ( ! is_array( $this->options_css ) ) {
			$this->options_css = array();
		}

	}

	/**
	 * Get css code extracted from theme options.
	 *
	 * @since   1.0.0
	 * @return  string
	 */
	public function get_css( $options_css_selectors ) {

		$css_arr = $css_arr_by_media = array();
		$css_properties = $this->css_properties;
		array_unshift( $css_properties, '' );

		foreach ( $options_css_selectors as $option_raw_key => $css_map ) {

			$selector = ( array_key_exists( 'selector', $css_map ) ? ( is_array( $css_map['selector'] ) ? implode( ', ', $css_map['selector'] ) : $css_map['selector'] ) : '' );
			$breakpoints = array_key_exists( 'breakpoints', $css_map ) ? $css_map['breakpoints'] : array();

			foreach ( $css_properties as $property ) {

				$p_opt_key = $option_raw_key;
				if ( ! empty( $property ) ) {
					$p_opt_key .= '_' . $property;
				}

				foreach ( $this->options_css as $option_key => $option_value ) {

					$option_key = self::parse_option_key( $option_key );

					if ( $p_opt_key == $option_key['key'] ) {

						if ( ! empty( $selector ) ) {

							$parsed_inline_css = $this->parse_css_inline( $property, $option_value, $selector );
							$css_arr_key = $option_raw_key . ( empty( $option_key['id_prefix'] ) ? '' : '@' . $option_key['id_prefix'] );
							if ( array_key_exists( 'breakpoints', $option_key ) ) {
								$css_arr_key .= '#' . $option_key['breakpoints'];
							}

							if ( ! array_key_exists( $css_arr_key, $css_arr ) ) {
								$css_arr[ $css_arr_key ] = array(
									'id_prefix'    => $option_key['id_prefix'],
									'selector'     => $parsed_inline_css['selector'],
									'declarations' => array(),
									'breakpoints'  => array_key_exists( 'breakpoints', $option_key ) ? array( $this->get_breakpoint_by_id( $option_key['breakpoints'] ) ) : $breakpoints
								);
							}

							$css_arr[ $css_arr_key ]['declarations'] = wp_parse_args( $parsed_inline_css['declarations'], $css_arr[ $css_arr_key ]['declarations'] );

						}

						if ( array_key_exists( 'children', $css_map ) && is_array( $css_map['children'] ) && is_array( $option_value ) ) {

							foreach ( $option_value as $k => $v ) {

								$k = self::parse_option_key( $k );

								if ( preg_match( '#^(.*?)_('. implode( '|', $this->css_properties ) .')$#', $k['key'], $match ) && array_key_exists( $match[1], $css_map['children'] ) && is_array( $css_map['children'][ $match[1] ] ) ) {

									if ( array_key_exists( 'selector', $css_map['children'][ $match[1] ] ) ) {
										$selector = is_array( $css_map['children'][ $match[1] ]['selector'] ) ? implode( ', ', $css_map['children'][ $match[1] ]['selector'] ) : $css_map['children'][ $match[1] ]['selector'];
									}

									if ( array_key_exists( 'breakpoints', $css_map['children'][ $match[1] ] ) ) {
										$breakpoints = $css_map['children'][ $match[1] ]['breakpoints'];
									}

									if ( ! empty( $selector ) ) {

										$parsed_inline_css = $this->parse_css_inline( $match[2], $v, $selector );
										$css_arr_key = $option_raw_key . '_' . $match[1] . ( empty( $k['id_prefix'] ) ? '' : '@' . $k['id_prefix'] );
										if ( array_key_exists( 'breakpoints', $option_key ) ) {
											$css_arr_key .= '#' . $option_key['breakpoints'];
										}

										if ( ! array_key_exists( $css_arr_key, $css_arr ) ) {
											$css_arr[ $css_arr_key ] = array(
												'id_prefix'    => $k['id_prefix'],
												'selector'     => $parsed_inline_css['selector'],
												'declarations' => array(),
												'breakpoints'  => array_key_exists( 'breakpoints', $k ) ? array( $this->get_breakpoint_by_id( $k['breakpoints'] ) ) : $breakpoints
											);
										}

										$css_arr[ $css_arr_key ]['declarations'] = wp_parse_args( $parsed_inline_css['declarations'], $css_arr[ $css_arr_key ]['declarations'] );

									}

								}

							}

						}

					}

				}

			}

		}

		foreach ( $css_arr as $k => $v ) {

			$v['selector'] = $this->parse_css_selector( $v['selector'], $v['id_prefix'] );

			if ( array_key_exists( 'breakpoints', $v )
				&& is_array( $v['breakpoints'] )
				&& ! empty( $v['breakpoints'] ) ) {

				foreach ( $v['breakpoints'] as $b ) {

					$media_key = ( array_key_exists( 'min', $b ) ? $b['min'] : '0' ) . '_' . ( array_key_exists( 'max', $b ) ? $b['max'] : '0' );

					if ( ! array_key_exists( $media_key, $css_arr_by_media ) ) {
						$css_arr_by_media[ $media_key ] = array();
					}

					if ( ! array_key_exists( 'properties', $b ) || empty( $b['properties'] ) ) {
						$b['properties'] = array_keys( $v['declarations'] );
					}

					foreach ( $v['selector'] as $selector ) {

						if ( ! array_key_exists( $selector, $css_arr_by_media[ $media_key ] ) ) {
							$css_arr_by_media[ $media_key ][ $selector ] = array();
						}

						foreach ( $b['properties'] as $p ) {
							if ( array_key_exists( $p, $v['declarations'] ) ) {
								$css_arr_by_media[ $media_key ][ $selector ][ $p ] = $v['declarations'][ $p ];
								unset( $v['declarations'][ $p ] );
							}
						}

					}

				}

			}

			if ( ! empty( $v['declarations'] ) ) {

				if ( ! array_key_exists( '0_0', $css_arr_by_media ) ) {
					$css_arr_by_media['0_0'] = array();
				}

				foreach ( $v['selector'] as $selector ) {

					if ( ! array_key_exists( $selector, $css_arr_by_media['0_0'] ) ) {
						$css_arr_by_media['0_0'][ $selector ] = array();
					}

					$css_arr_by_media['0_0'][ $selector ] = wp_parse_args( $v['declarations'], $css_arr_by_media['0_0'][ $selector ] );

				}

			}

		}

		ksort( $css_arr_by_media );

		$output_arr = array();

		foreach ( $css_arr_by_media as $s => $c ) {

			list( $min_width, $max_width ) = explode( '_', $s );

			if ( $s == '0_0' ) {
				$media_str = 'all';
			} else {

				$media_str = '@media only screen';

				if ( $min_width != '0' ) {
					$media_str .= ' and (min-width:'. $min_width .'px)';
				}

				if ( $max_width != '0' ) {
					$media_str .= ' and (max-width:'. $max_width .'px)';
				}

			}

			$output_arr[ $media_str ] = array();

			foreach ( $c as $i => $j ) {
				if ( ! empty( $j ) ) {
					$output_arr[ $media_str ][ $i ] = $this->shorten_css_declarations( $j );
				}
			}

		}

		$output = '';

		foreach ( $output_arr as $media_key => $css_blocks ) {

			if ( $media_key != 'all' ) {
				$output .= $media_key . '{';
			}

			foreach( $css_blocks as $selector => $declatarions ) {

				$output .= $selector . '{';
				foreach ( $declatarions as $property => $value ) {
					$output .= $property .':'. $value .';';
				}
				$output = rtrim( $output, ';' ) . '}';

			}

			if ( $media_key != 'all' ) {
				$output .= '}';
			}

		}

		return $output;

	}

	/**
	 * Parse css selector.
	 *
	 * @since   1.0.0
	 * @param   string $selector
	 * @param   string $id_prefix
	 * @return  array
	 */
	private function parse_css_selector( $selector, $id_prefix = '' ) {

		if ( ! is_array( $selector ) ) {
			$selector = is_string( $selector ) ? array_map( function( $v ) { return trim( $v ); }, explode( ',', $selector ) ) : array();
		}

		$_selector = array();
		$i = 0;

		foreach ( $selector as $s ) {

			if ( false !== strpos( $s, ':-' ) ) {
				$i++;
			}

			if ( ! array_key_exists( $i, $_selector ) ) {
				$_selector[ $i ] = '';
			}

			$_selector[ $i ] .= ',' . ( empty( $id_prefix ) ? '' : '#' . $id_prefix . ' ' ) . str_replace( ' > ', '>', $s );

		}

		return array_map( function( $v ) { return ltrim( $v, ',' ); }, $_selector );

	}

	/**
	 * Extract id_prefix and breakpoint id from given option key.
	 *
	 * @since   1.0.0
	 * @param   string $key
	 * @return  array
	 */
	public static function parse_option_key( $key ) {

		$result = array(
			'key'       => '',
			'id_prefix' => ''
		);

		if ( ! is_string( $key ) ) {
			$key = '';
		}

		$key_exp = array_map( function( $v ) { return explode( '#', $v ); }, explode( '@', $key ) );

		$result['key'] = $key_exp[0][0];

		if ( count( $key_exp[0] ) > 1 ) {
			$result['breakpoints'] = $key_exp[0][1];
		}

		if ( count( $key_exp ) > 1 ) {
			$result['id_prefix'] = $key_exp[1][0];
			if ( count( $key_exp[1] ) > 1 ) {
				$result['breakpoints'] = $key_exp[1][1];
			}
		}

		return $result;

	}

	/**
	 * Collect css declarations for any given option.
	 *
	 * @since   1.0.0
	 * @param   string $property
	 * @param   mixed $value
	 * @param   string $selector
	 * @return  array
	 */
	private function parse_css_inline( $property, $value, $selector = '' ) {

		$result = array(
			'declarations' => array()
		);

		if ( preg_match( '#^(margin|padding)(?:_(x|y|top|right|bottom|left))?$#', $property, $match ) ) {

			$prop_arr = array( $match[1] );

			if ( count( $match ) > 2 ) {
				switch ( $match[2] ) {
					case 'x':
						$prop_arr[] = $prop_arr[0] . '-left';
						$prop_arr[0] .= '-right';
						break;
					case 'y':
						$prop_arr[] = $prop_arr[0] . '-top';
						$prop_arr[0] .= '-bottom';
						break;
					default:
						$prop_arr[0] .=  '-' . $match[2];
						break;
				}
			}

			if ( is_numeric( $value ) ) {
				foreach ( $prop_arr as $prop ) {
					$result['declarations'][ $prop ] = $value . 'px';
				}
			}

		} else if ( preg_match( '#^(border)(?:_(top|right|bottom|left))?_(color|width)$#', $property, $match ) ) {

			$match = array_filter( $match );
			array_shift( $match );

			if ( is_numeric( $value ) || $this->is_color( $value ) ) {
				$result['declarations'][ implode( '-', $match ) ] = $value . ( end( $match ) == 'width' ? 'px' : '' );
			}

		} else if ( preg_match( '#^(width|height|font_size|line_height)$#', $property ) ) {

			if ( is_numeric( $value ) ) {
				$result['declarations'][ str_replace( '_', '-', $property ) ] = $value . 'px';
			} else if ( is_array( $value ) && array_key_exists( 'font_size', $value ) ) {
				$result['declarations']['font-size'] = $value['font_size'] . 'px';
			}

		} else if ( $property == 'font_weight' ) {

			if ( is_numeric( $value ) ) {
				$result['declarations'][ str_replace( '_', '-', $property ) ] = $value;
			} else if ( is_array( $value ) && array_key_exists( 'font_weight_style', $value ) && preg_match( '#^([0-9]{3})(italic)?$#', $value['font_weight_style'], $match ) ) {
				$result['declarations']['font-weight'] = $match[1];
				if ( count( $match ) > 2 ) {
					$result['declarations']['font-style'] = $match[2];
				}
			}

		} else if ( $property == 'text_transform' ) {

			if ( in_array( $value, array( 'none', 'uppercase', 'lowercase', 'capitalize' ) ) ) {
				$result['declarations']['text-transform'] = $value;
			}

		} else if ( preg_match( '#^(color|background_color)$#', $property ) ) {

			if ( $this->is_color( $value ) ) {
				$result['declarations'][ str_replace( '_', '-', $property ) ] = $value;
			}

		} else if ( $property == 'font' ) {

			if ( is_array( $value ) ) {

				$value = wp_parse_args( $value, array(
					'font_family'        => '',
					'backup_font_family' => '',
					'font_weight_style'  => '',
					'font_subsets'       => '',
					'font_size'          => ''
				) );

				if ( ! empty( $value['font_family'] ) ) {
					$is_google_font = ( false === strpos( $value['font_family'], ',' ) );
					$result['declarations']['font-family'] = ( $is_google_font ? '"' : '' ) . $value['font_family'] . ( $is_google_font ? '"' : '' ) . ( $is_google_font ? ( ', ' . ( empty( $value['backup_font_family'] ) ? 'sans-serif' : $value['backup_font_family'] ) ) : '' );
				}

				if ( preg_match( '#^([0-9]{3})(italic)?$#', $value['font_weight_style'], $match ) ) {
					$result['declarations']['font-weight'] = $match[1];
					if ( count( $match ) > 2 ) {
						$result['declarations']['font-style'] = $match[2];
					}
				}

				if ( is_numeric( $value['font_size'] ) ) {
					$result['declarations']['font-size'] = $value['font_size'] . 'px';
				}

			}

		} else if ( $property == 'background' ) {

			if ( is_array( $value ) ) {

				$value = wp_parse_args( $value, array(
					'background_color'      => '',
					'background_image'      => '',
					'background_repeat'     => '',
					'background_position'   => '',
					'background_attachment' => ''
				) );

				$result['declarations']['background'] = '';

				if ( ! empty( $value['background_image'] ) ) {

					$image = wp_get_attachment_image_src( $value['background_image'], 'full' );

					if ( $image ) {

						$result['declarations']['background'] .= 'url('. esc_url( $image[0] ) .') ';

						if ( ! empty( $value['background_position'] ) ) {
							$result['declarations']['background'] .= str_replace( '_', ' ', $value['background_position'] ) .'/';
						}

						if ( ! empty( $value['background_size'] ) ) {
							$result['declarations']['background'] .= $value['background_size'] . ' ';
						} else {
							$result['declarations']['background'] = rtrim( $result['declarations']['background'], '/' ) .' ';
						}

						if ( ! empty( $value['background_repeat'] ) ) {
							$result['declarations']['background'] .= str_replace( '_', '-', $value['background_repeat'] ) .' ';
						}

						if ( ! empty( $value['background_attachment'] ) ) {

							if ( $value['background_attachment'] != 'fixed' ) {
								$result['declarations']['background'] .= $value['background_attachment'] .' ';
							}

						}

					}

				}

				if ( $this->is_color( $value['background_color'] ) ) {
					$result['declarations']['background'] .= $value['background_color'] .' ';
				}

				$result['declarations']['background'] = rtrim( $result['declarations']['background'] );

				if ( empty( $result['declarations']['background'] ) ) {
					unset( $result['declarations']['background'] );
				}

			}

		}

		$result['selector'] = $selector;

		return $result;

	}


	/**
	 * Shorten given css properties.
	 *
	 * @since   1.0.0
	 * @param   array $declarations
	 * @return  array
	 */
	private function shorten_css_declarations( $declarations = array() ) {

		if ( ! is_array( $declarations ) ) {
			return array();
		}

		if ( array_key_exists( 'font-size', $declarations ) && array_key_exists( 'font-family', $declarations ) ) {

			$declarations['font'] = '';

			if ( array_key_exists( 'font-style', $declarations ) ) {
				$declarations['font'] .= $declarations['font-style'] . ' ';
				unset( $declarations['font-style'] );
			}

			if ( array_key_exists( 'font-weight', $declarations ) ) {
				$declarations['font'] .= $declarations['font-weight'] . ' ';
				unset( $declarations['font-weight'] );
			}

			$declarations['font'] .= $declarations['font-size'] . ' ';

			if ( array_key_exists( 'line-height', $declarations ) ) {
				$declarations['font'] = rtrim( $declarations['font'] ) . '/' . $declarations['line-height'] . ' ';
				unset( $declarations['line-height'] );
			}

			$declarations['font'] .= $declarations['font-family'];

			unset( $declarations['font-size'], $declarations['font-family'] );

		} else if ( array_key_exists( 'padding-top', $declarations ) && array_key_exists( 'padding-right', $declarations) && array_key_exists( 'padding-bottom', $declarations) && array_key_exists( 'padding-left', $declarations) ) {

			$declarations['padding'] = $declarations['padding-top'] .' ';
			if ( $declarations['padding-top'] != $declarations['padding-right'] ) {
				if ( $declarations['padding-top'] != $declarations['padding-bottom'] ) {
					$declarations['padding'] .= $declarations['padding-right'] .' '. $declarations['padding-bottom'] .' '. ( $declarations['padding-right'] != $declarations['padding-left'] ? $declarations['padding-left'] : '' );
				} else if ( $declarations['padding-right'] == $declarations['padding-left'] ) {
					$declarations['padding'] .= $declarations['padding-right'];
				}
			}

			$declarations['padding'] = rtrim( $declarations['padding'] );

			unset( $declarations['padding-top'], $declarations['padding-right'], $declarations['padding-bottom'], $declarations['padding-left'] );

		} else if ( array_key_exists( 'margin-top', $declarations ) && array_key_exists( 'margin-right', $declarations) && array_key_exists( 'margin-bottom', $declarations) && array_key_exists( 'margin-left', $declarations) ) {

			$declarations['margin'] = $declarations['margin-top'] .' ';
			if ( $declarations['margin-top'] != $declarations['margin-right'] ) {
				if ( $declarations['margin-top'] != $declarations['margin-bottom'] ) {
					$declarations['margin'] .= $declarations['margin-right'] .' '. $declarations['margin-bottom'] .' '. ( $declarations['margin-right'] != $declarations['margin-left'] ? $declarations['margin-left'] : '' );
				} else if ( $declarations['margin-right'] == $declarations['margin-left'] ) {
					$declarations['margin'] .= $declarations['margin-right'];
				}
			}

			$declarations['margin'] = rtrim( $declarations['margin'] );

			unset( $declarations['margin-top'], $declarations['margin-right'], $declarations['margin-bottom'], $declarations['margin-left'] );

		}

		foreach ( $declarations as $property => $value ) {
			if ( preg_match( '#^\#([0-9a-f]{6})$#', $value, $match ) ) {

				$shorten_hex = '';
				$color_exp = str_split( $match[1], 2 );
				foreach ( $color_exp as $j ) {
					if ( count( array_unique( str_split( $j ) ) ) == 1 ) {
						$shorten_hex .= substr( $j, 0, 1 );
					}
				}

				$declarations[ $property ] = ( strlen( $shorten_hex ) == 3 ) ? '#' . $shorten_hex : $value;

			}
		}

		return $declarations;

	}

	/**
	 * Get breakpoint by given id
	 *
	 * @since   1.0.0
	 * @param   string $breakpoint_id
	 * @return  array
	 */
	private function get_breakpoint_by_id( $breakpoint_id ) {

		$breakpoint_map = array(
			'xs' => array(
				'max' => 767
			),
			'sm' => array(
				'min' => 768,
				'max' => 991
			),
			'md' => array(
				'min' => 992,
				'max' => 1199
			),
			'lg' => array(
				'min' => 1200
			)
		);

		if ( ! is_string( $breakpoint_id ) || ! array_key_exists( $breakpoint_id, $breakpoint_map ) ) {
			return array();
		}

		return $breakpoint_map[ $breakpoint_id ];

	}

	/**
	 * Get google fonts url extracted from theme options.
	 *
	 * @since   1.0.0
	 * @param   array $fonts
	 * @return  string
	 */
	public function get_fonts_url( $fonts = array() ) {

		if ( ! is_array( $fonts ) ) {
			$fonts = array();
		}

		if ( array_key_exists( 'module', $this->options_css ) ) {
			foreach ( $this->options_css['module'] as $k => $v ) {
				$this->options_css['module_' . $k ] = $v;
			}
			unset( $this->options_css['module'] );
		}

		foreach ( $this->options_css as $k => $v ) {
			if ( preg_match( '#^(.*?)_font(\@(.*?))?$#', $k, $match ) ) {
				$fonts[] = $v;
			}
		}

		$_fonts = $font_subsets = array();

		foreach ( $fonts as $f ) {

			if ( ! is_array( $f ) || ! array_key_exists( 'font_family', $f ) || false !== strpos( $f['font_family'], ',' ) ) {
				continue;
			}

			$font_family = str_replace( ' ', '+', $f['font_family'] );

			if ( ! array_key_exists( $font_family, $_fonts ) ) {
				$_fonts[ $font_family ] = array();
			}

			if ( array_key_exists( 'font_weight_style', $f ) ) {

				if ( ! is_array( $f['font_weight_style'] ) ) {
					$f['font_weight_style'] = array( $f['font_weight_style'] );
				}

				foreach ( $f['font_weight_style'] as $w ) {
					if ( ! in_array( $w, $_fonts[ $font_family ] ) ) {
						$_fonts[ $font_family ][] = $w;
					}
				}

			}

			if ( array_key_exists( 'font_subsets', $f ) ) {

				if ( ! is_array( $f['font_subsets'] ) ) {
					$f['font_subsets'] = array( $f['font_subsets'] );
				}

				foreach ( $f['font_subsets'] as $s ) {
					if ( ! in_array( $s, $font_subsets ) ) {
						$font_subsets[] = $s;
					}
				}

			}

		}

		if ( empty( $_fonts ) ) {
			return '';
		}

		$url_font_family = array();
		foreach ( $_fonts as $k => $v ) {
			if ( empty( $v ) ) {
				$v = array( '400' );
			}
			$url_font_family[] = $k .':'. implode( ',', $v );
		}

		return '//fonts.googleapis.com/css?family='. implode( '|', $url_font_family ) .'&subset='. implode( ',', $font_subsets );

	}

	/**
	 * Check if given string is color.
	 *
	 * @since   1.0.0
	 * @param   $str
	 * @return  bool
	 */
	public function is_color( $str ) {

		return ( is_string( $str ) && preg_match( '#^(?:(rgba?)\(\s*([0-9]{1,3})\s*\,\s*([0-9]{1,3})\s*\,\s*([0-9]{1,3})\s*(?:\,\s*((?:0\.*(?:[0-9]{1,2})*)|1)\s*)*\)|(\#[a-f0-9]{6}))$#i', $str ) );

	}

}